/**
 * @file
 * Template System - Template rendering and helper functions
 */

(function (Drupal) {
  'use strict';

  /**
   * Template System
   */
  window.BusinessMapTemplateSystem = {
    // Category to icon mapping
    categoryIcons: {
      'restaurant': '<i class="fas fa-utensils"></i>',
      'shopping': '<i class="fas fa-shopping-bag"></i>',
      'culture': '<i class="fas fa-landmark"></i>',
      'hotel': '<i class="fas fa-hotel"></i>',
      'park': '<i class="fas fa-tree"></i>',
      'museum': '<i class="fas fa-museum"></i>',
      'cafe': '<i class="fas fa-coffee"></i>',
      'bar': '<i class="fas fa-glass-cheers"></i>'
    },

    // Category to name mapping
    categoryNames: {
      'restaurant': Drupal.t('Restaurants & Cafés'),
      'shopping': Drupal.t('Shopping'),
      'culture': Drupal.t('Cultural Sites'),
      'hotel': Drupal.t('Accommodation'),
      'park': Drupal.t('Parks & Recreation'),
      'museum': Drupal.t('Museums'),
      'cafe': Drupal.t('Cafés'),
      'bar': Drupal.t('Bars & Lounges')
    },

    // Country code to name mapping
    countries: {
      'CH': Drupal.t('Switzerland'),
      'IT': Drupal.t('Italy'),
      'US': Drupal.t('United States'),
      'GB': Drupal.t('United Kingdom'),
      'DE': Drupal.t('Germany')
    },

    /**
     * Render a template with data
     */
    render: function(templateName, data = {}) {
      const template = this.templates[templateName];
      if (typeof template === 'function') {
        try {
          return template.call(this, data);
        } catch (error) {
          console.error('[Template System] Error rendering template:', error);
          return this.errorTemplate(templateName);
        }
      }
      return this.errorTemplate(templateName);
    },

    /**
     * Error template
     */
    errorTemplate: function(templateName) {
      return `<div class="error-template">${Drupal.t('Template "@name" not found', {
        '@name': templateName
      })}</div>`;
    },

    /**
     * Helper: Get country name from code
     */
    getCountryName: function(countryCode) {
      return this.countries[countryCode] || countryCode || '';
    },

    /**
     * Helper: Get category icon
     */
    getCategoryIcon: function(category) {
      return this.categoryIcons[category] || '<i class="fas fa-star"></i>';
    },

    /**
     * Helper: Get category name
     */
    getCategoryName: function(category) {
      return this.categoryNames[category] || Drupal.t(category);
    },

    /**
     * Helper: Format address lines
     */
    formatAddress: function(address) {
      const lines = [];
      if (address.street) lines.push(address.street);
      if (address.street2) lines.push(address.street2);

      const cityLine = [address.postalCode, address.city]
        .filter(Boolean)
        .join(' ');
      if (cityLine) lines.push(cityLine);

      if (address.state) lines.push(address.state);
      if (address.country) lines.push(this.getCountryName(address.country));

      return lines.join('<br>');
    },

    /**
     * Template: Setup Incomplete Warning
     */
    setupWarningTemplate: function(missingConfigs) {
      if (!missingConfigs || missingConfigs.length === 0) {
        return '';
      }

      const itemsHtml = missingConfigs.map(config => `
        <li>
          <a href="${config.path}" target="_blank" class="setup-link">
            <i class="fas fa-exclamation-circle"></i>
            <strong>${config.name}:</strong> ${config.description}
          </a>
        </li>
      `).join('');

      return `
        <div class="setup-warning">
          <div class="setup-warning-header">
            <i class="fas fa-exclamation-triangle"></i>
            <h3>${Drupal.t('Setup Incomplete')}</h3>
          </div>
          <div class="setup-warning-content">
            <p>${Drupal.t('Please complete the following configurations to fully use the map features:')}</p>
            <ul class="setup-links">
              ${itemsHtml}
            </ul>
            <div class="setup-note">
              <p><small><i class="fas fa-info-circle"></i> ${Drupal.t('These links will open in a new tab. After saving changes, refresh this page.')}</small></p>
            </div>
          </div>
        </div>
      `;
    },

    /**
     * All templates
     */
    templates: {
      where: function(data) {
        const address = data.address || {};
        const formattedAddress = this.formatAddress(address);

        return `
          <div class="location-info">
            <p>${Drupal.t('Our headquarters is located in @city, offering easy access and modern facilities.', {
              '@city': address.city || ''
            })}</p>
            <div class="contact-details">
              <p><strong>📍 ${Drupal.t('Address:')}</strong><br>${formattedAddress}</p>
              ${address.notes ? `<p><strong>📋 ${Drupal.t('Access Notes:')}</strong> ${address.notes}</p>` : ''}
              <p><strong>📞 ${Drupal.t('Phone:')}</strong> ${address.country}</p>
              <p><strong>✉️ ${Drupal.t('Email:')}</strong> ${address.city}</p>
              <p><strong>🕒 ${Drupal.t('Hours:')}</strong><br>${address.country}</p>
            </div>
          </div>
        `;
      },

      car: function(data) {
        const transportation = data.transportation || {};
        const highwayExit = transportation.highwayExit || {};

        return `
          <div class="transport-info">
            <h3><strong>${Drupal.t('Driving Directions:')}</strong></h3>
            ${highwayExit.name ? `<h4>${highwayExit.name}</h4>` : ''}
            ${highwayExit.description ? `<p>${highwayExit.description}</p>` : ''}
            ${transportation.carRental ? `
              <div class="car-rental-info">
                <h4><i class="fas fa-car"></i> ${Drupal.t('Car Rental:')}</h4>
                <p>${transportation.carRental}</p>
              </div>
            ` : ''}
          </div>
        `;
      },

      publicTransport: function(data) {
        const transportation = data.transportation || {};
        const transportOptions = transportation.transportModeOptions || {};

        let optionsHtml = '';
        if (Object.keys(transportOptions).length > 0) {
          optionsHtml = `
            <p><strong>${Drupal.t('Available Transport Modes:')}</strong></p>
            <ul>
              ${Object.values(transportOptions).map(option =>
                `<li>${Drupal.t(option)}</li>`
              ).join('')}
            </ul>
          `;
        }

        const airportHtml = transportation.airport ? `
          <div class="airport-info">
            <h4><i class="fas fa-plane"></i> ${Drupal.t('Nearest Airport:')}</h4>
            <p><strong>${transportation.airport.name || ''}</strong></p>
            ${transportation.airport.description ?
              `<p>${transportation.airport.description}</p>` : ''}
          </div>
        ` : '';

        const busHtml = (transportation.busStation && transportation.busStation.enabled) ? `
          <div class="bus-station-info">
            <h4><i class="fas fa-bus"></i> ${Drupal.t('Bus Station:')}</h4>
            <p>${transportation.busStation.name || ''}</p>
          </div>
        ` : '';

        const defaultTransport = transportation.defaultTransportMode ? `
          <div class="default-transport">
            <p><strong>${Drupal.t('Recommended Transport:')}</strong>
              ${transportOptions[transportation.defaultTransportMode] || ''}
            </p>
          </div>
        ` : '';

        return `
          <div class="transport-info">
            ${optionsHtml}
            ${airportHtml}
            ${busHtml}
            ${defaultTransport}
          </div>
        `;
      },

      parking: function(data) {
        const parking = data.parking || {};
        const capacity = parking.capacity || {};
        const parkingZones = parking.parkingZones || [];

        // Capacity list
        let capacityHtml = '';
        if (capacity.total) {
          capacityHtml = `
            <p><strong>${Drupal.t('Parking Capacity:')}</strong></p>
            <ul>
              <li><strong>${Drupal.t('Total Spaces:')}</strong> ${capacity.total}</li>
              ${capacity.disabled ? `<li><strong>${Drupal.t('Disabled Spaces:')}</strong> ${capacity.disabled}</li>` : ''}
              ${capacity.family ? `<li><strong>${Drupal.t('Family Spaces:')}</strong> ${capacity.family}</li>` : ''}
              ${capacity.ev ? `<li><strong>${Drupal.t('EV Charging:')}</strong> ${capacity.ev}</li>` : ''}
            </ul>
          `;
        }

        // Parking zones
        let zonesHtml = '';
        if (parkingZones.length > 0) {
          zonesHtml = `
            <div class="parking-zones">
              <h4>${Drupal.t('Parking Zones:')}</h4>
              ${parkingZones.map(zone => `
                <div class="parking-zone">
                  <h4>${zone.name || ''}</h4>
                  ${zone.description ? `<p>${zone.description}</p>` : ''}
                  ${zone.spaces ? `<p><strong>${Drupal.t('Spaces:')}</strong> ${zone.spaces}</p>` : ''}
                </div>
              `).join('')}
            </div>
          `;
        }

        // Main parking
        const mainParkingHtml = parking.mainParking ? `
          <div class="main-parking">
            <h4>${Drupal.t('Main Parking:')}</h4>
            <p><strong>${parking.mainParking.name || ''}</strong></p>
            ${parking.mainParking.description ?
              `<p>${parking.mainParking.description}</p>` : ''}
          </div>
        ` : '';

        return `
          <div class="parking-info">
            ${capacityHtml}
            ${parking.hours ? `<p><strong>${Drupal.t('Hours:')}</strong> ${parking.hours}</p>` : ''}
            ${parking.feeType ? `
              <p><strong>${Drupal.t('Fee Type:')}</strong>
                ${parking.parkingFeeTypeOptions && parking.parkingFeeTypeOptions[parking.feeType] ?
                  Drupal.t(parking.parkingFeeTypeOptions[parking.feeType]) :
                  Drupal.t(parking.feeType)}
              </p>
            ` : ''}
            ${zonesHtml}
            ${mainParkingHtml}
          </div>
        `;
      },

      // attractions: function(data) {
      //   const nearbyMarkers = data.nearbyMarkers || [];
      //
      //   if (nearbyMarkers.length === 0) {
      //     return `
      //       <div class="attractions-info">
      //         <p>${Drupal.t('No nearby attractions configured.')}</p>
      //       </div>
      //     `;
      //   }


        attractions: function(data) {
          const nearbyMarkers = data.nearbyMarkers || [];

          if (nearbyMarkers.length === 0) {
            return `
              <div class="attractions-info">
                <p>${Drupal.t('No nearby attractions configured.')}</p>
                <p>
                  <a href="/admin/config/map/settings" class="button button--small" style="margin-top: 5px;">
                    ${Drupal.t('Map Settings > Nearby')}
                  </a>
                </p>
              </div>
            `;
          }


        // Group by category
        const categories = {};
        nearbyMarkers.forEach(marker => {
          if (marker.category && marker.enabled !== false) {
            if (!categories[marker.category]) {
              categories[marker.category] = [];
            }
            categories[marker.category].push(marker);
          }
        });

        // Generate HTML for each category
        const categoriesHtml = Object.entries(categories).map(([category, items]) => {
          const icon = this.getCategoryIcon(category);
          const categoryName = this.getCategoryName(category);

          const itemsHtml = items.map(item => `
            <li>
              <strong>${item.name || ''}</strong>
              ${item.description ? `- ${item.description}` : ''}
              ${item.distance ? `<br><small>(${item.distance})</small>` : ''}
            </li>
          `).join('');

          return `
            <div class="attraction-category">
              <h4>${icon} ${categoryName}</h4>
              <ul>${itemsHtml}</ul>
            </div>
          `;
        }).join('');

        return `
          <div class="attractions-info">
            <p><strong>${Drupal.t('Discover what\'s around our location:')}</strong></p>
            ${categoriesHtml}
          </div>
        `;
      },

      myLocation: function(data) {
        const position = data.position || {};

        return `
          <div class="location-info">
            <p><strong>${Drupal.t('Your current location has been found!')}</strong></p>
            <p>${Drupal.t('Latitude:')} ${position.latitude ? position.latitude.toFixed(6) : ''}</p>
            <p>${Drupal.t('Longitude:')} ${position.longitude ? position.longitude.toFixed(6) : ''}</p>
            ${position.accuracy ? `
              <p>${Drupal.t('Accuracy:')} ${Math.round(position.accuracy)} ${Drupal.t('meters')}</p>
            ` : ''}
            <p>${Drupal.t('Click "Route to Business" on the map marker to get directions.')}</p>
          </div>
        `;
      }
    }
  };

})(Drupal);
