/**
 * @file
 * UI Manager - Handles user interface interactions and display
 */

(function() {
  'use strict';

  const UIManager = {
    mapManager: null,
    locations: {
      where: {
        title: "",
        content: "",
        center: [],
        zoom: 14,
        markers: [
          {coords: [], type: 'main', label: ""}
        ]
      },
      car: {
        title: "",
        content: "",
        center: [],
        zoom: 12,
        markers: [
          {coords: [], type: 'main', label: ""},
          {coords: [], type: 'toll', label: ""}
        ]
      },
      publicTransport: {
        title: "Public Transport",
        content: "",
        center: [41.8349, 12.4700],
        zoom: 14,
        markers: [
          {coords: [], type: 'main', label: ""},
          {coords: [], type: 'train', label: ""},
          {coords: [], type: 'bus', label: ""},
          {coords: [], type: 'subway', label: ""}
        ]
      }
    },

    /**
     * Initialize UI Manager
     */
    initialize: function() {
      console.log('Initializing UI Manager...');
      this.mapManager = window.MapManager;
      console.log('UI Manager initialized');
    },

    /**
     * Show category with markers
     */
    showCategory: function(category) {
      console.log(`Showing category: ${category}`);
      
      // Remove routes if any
      if (window.RouteManager) {
        window.RouteManager.removeAllRoutes();
      }

      // Hide user location notification
      const userLocationNotification = document.getElementById('userLocationNotification');
      if (userLocationNotification) {
        userLocationNotification.classList.add('hidden');
      }

      // Update button states
      this.updateButtonStates(category);

      const data = this.locations[category];
      if (!data) {
        console.error(`Category ${category} not found`);
        return;
      }

      // Update info panel
      this.updateInfoPanel(data.title, data.content);

      // Fly to location
      this.mapManager.flyToLocation(data.center, data.zoom);

      // Clear existing markers and add new ones
      this.mapManager.clearMarkers();
      this.addCategoryMarkers(data.markers);

      // Show routes if needed
      this.showCategoryRoutes(category);
    },

    /**
     * Update navigation button states
     */
    updateButtonStates: function(activeCategory) {
      const navButtons = document.querySelectorAll('.nav-button');
      const locationBtn = document.getElementById('locationBtn');

      if (locationBtn && activeCategory !== 'mylocation') {
        locationBtn.classList.remove('active');
      }

      navButtons.forEach(btn => {
        if (btn.dataset.target === activeCategory) {
          btn.classList.add('active');
        } else if (btn.id !== 'locationBtn') {
          btn.classList.remove('active');
        }
      });
    },

    /**
     * Update info panel content
     */
    updateInfoPanel: function(title, content) {
      const infoTitle = document.getElementById('infoTitle');
      const infoContent = document.getElementById('infoContent');
      const infoPanel = document.getElementById('infoPanel');

      if (infoTitle) infoTitle.textContent = title;
      if (infoContent) infoContent.innerHTML = content;
      if (infoPanel) infoPanel.classList.remove('hidden');
    },

    /**
     * Add markers for a category
     */
    addCategoryMarkers: function(markers) {
      markers.forEach((location, index) => {
        setTimeout(() => {
          this.mapManager.addMarker(location.coords, location.type, location.label);
        }, index * 100);
      });
    },

    /**
     * Show routes for specific category
     */
    showCategoryRoutes: function(category) {
      if (!window.RouteManager) return;

      setTimeout(() => {
        if (category === 'car') {
          window.RouteManager.showCarRoute();
        } else if (category === 'publicTransport') {
          window.RouteManager.showPublicTransportRoute();
        }
      }, 800);
    },

    /**
     * Show notification
     */
    showNotification: function(message, type = 'info') {
      // Remove existing notifications
      document.querySelectorAll('.notification').forEach(n => n.remove());

      const notification = document.createElement('div');
      notification.className = `notification ${type}`;
      notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
        <span>${message}</span>
      `;

      document.body.appendChild(notification);

      setTimeout(() => {
        if (notification.parentNode) {
          notification.remove();
        }
      }, 4000);
    },

    /**
     * Show loading state for location button
     */
    showLocationLoading: function() {
      const locationBtn = document.getElementById('locationBtn');
      if (!locationBtn) return;

      this.locationButtonOriginalHTML = locationBtn.innerHTML;
      locationBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Locating...</span>';
      locationBtn.disabled = true;
    },

    /**
     * Reset location button state
     */
    resetLocationButton: function() {
      const locationBtn = document.getElementById('locationBtn');
      if (!locationBtn || !this.locationButtonOriginalHTML) return;

      locationBtn.innerHTML = this.locationButtonOriginalHTML;
      locationBtn.disabled = false;
    },

    /**
     * Activate location button
     */
    activateLocationButton: function() {
      const navButtons = document.querySelectorAll('.nav-button');
      navButtons.forEach(btn => {
        if (btn.id === 'locationBtn') {
          btn.classList.add('active');
        } else {
          btn.classList.remove('active');
        }
      });
    }
  };

  // Export to global scope
  window.UIManager = UIManager;

})();
