<?php

namespace Drupal\map\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\views\Views; // Add this line
use Symfony\Component\DependencyInjection\ContainerInterface;

// Add this if you want to use the GeolocationItem type check
use Drupal\geolocation\Plugin\Field\FieldType\GeolocationItem;
use Drupal\geofield\WktGeneratorInterface;
use Drupal\geofield\GeoPHP\GeoPHPInterface;



/**
 * Controller for map page.
 */
class MapPageController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The site settings.
   *
   * @var \Drupal\Core\Site\Settings
   */
  protected $siteSettings;

  /**
   * The module extension list.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected $moduleExtensionList;

  /**
   * Constructs a MapPageController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Site\Settings $site_settings
   *   The site settings.
   * @param \Drupal\Core\Extension\ModuleExtensionList $module_extension_list
   *   The module extension list.
   */
  public function __construct(ConfigFactoryInterface $config_factory, Settings $site_settings, ModuleExtensionList $module_extension_list) {
    $this->configFactory = $config_factory;
    $this->siteSettings = $site_settings;
    $this->moduleExtensionList = $module_extension_list;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('settings'),
      $container->get('extension.list.module')
    );
  }

  /**
   * Renders the map page.
   *
   * @return array
   *   A render array.
   */
  public function page() {
    $data = [];

    $data['config'] = $this->configFactory->get('map.settings');
    $data['site_config'] = $this->configFactory->get('system.site');

    // Get site coordinates - check if we should use site settings.
    if ($data['config']->get('use_site_coordinates')) {
      // Try to get coordinates from site settings module.
      $site_coords = $this->siteSettings->get('site_coordinates');

      if (!empty($site_coords) && is_array($site_coords)) {
        $center_lat = $site_coords['lat'] ?? $data['config']->get('center_lat');
        $center_lng = $site_coords['lng'] ?? $data['config']->get('center_lng');
      }
      else {
        // Fallback to module configuration.
        $center_lat = $data['config']->get('center_lat');
        $center_lng = $data['config']->get('center_lng');
      }
    }
    else {
      // Use module configuration.
      $center_lat = $data['config']->get('center_lat');
      $center_lng = $data['config']->get('center_lng');
    }


      // dump ($data['config']);
      // exit;

    // Prepare comprehensive configuration for JavaScript.
    $map_config = [
      // General settings.
      'center' => [
        (float) ($data['config']->get('center_lat') ?? 41.9028),
        (float) ($data['config']->get('center_lng') ?? 12.4964),
      ],
      'zoom' => (int) ($data['config']->get('default_zoom') ?? 12),
      'minZoom' => (int) ($data['config']->get('min_zoom') ?? 3),
      'maxZoom' => (int) ($data['config']->get('max_zoom') ?? 18),
      'zoomControl' => (bool) ($data['config']->get('zoom_control') ?? TRUE),
      'zoomControlPosition' => $data['config']->get('zoom_control_position') ?? 'topleft',
      'attributionControl' => (bool) ($data['config']->get('attribution_control') ?? TRUE),
      'mapType' => $data['config']->get('map_type') ?? 'openstreetmap',
      'useSiteCoordinates' => (bool) ($data['config']->get('use_site_coordinates') ?? FALSE),

      // Address settings.
      'address' => [
        'street' => $data['config']->get('street_address') ?? '',
        'street2' => $data['config']->get('street_address2') ?? '',
        'city' => $data['config']->get('city') ?? '',
        'state' => $data['config']->get('state_province') ?? '',
        'postalCode' => $data['config']->get('postal_code') ?? '',
        'country' => $data['config']->get('country') ?? 'CH',
        'notes' => $data['config']->get('address_note') ?? '',
        'format' => $data['config']->get('address_format') ?? 'multi_line',
        'formatted' => $this->formatAddressForConfig($data['config']),
      ],

      // Transportation settings.
      'transportation' => [
        'highwayExit' => [
          'enabled' => (bool) ($data['config']->get('highway_exit_enabled') ?? FALSE),
          'name' => $data['config']->get('highway_exit_name') ?? 'Meyrin Exit',
          'description' => $data['config']->get('highway_exit_description') ?? 'EMBEDDED - From Geneva A1/E25 highway, take exit 6 (Meyrin). Follow signs to CERN.',
          'lat' => (float) ($data['config']->get('highway_exit_lat') ?? 46.225),
          'lng' => (float) ($data['config']->get('highway_exit_lng') ?? 6.08),
        ],
        'trainStation' => [
          'enabled' => (bool) ($data['config']->get('train_station_enabled') ?? FALSE),
          'name' => $data['config']->get('train_station_name') ?? 'CERN Tram Stop (Meyrin-Gravière)',
          'description' => $data['config']->get('train_station_description') ?? 'Tram line 18: Direction CERN. Stops directly at CERN entrance. From Geneva main station, take tram 18.',
          'lat' => (float) ($data['config']->get('train_station_lat') ?? 46.2328),
          'lng' => (float) ($data['config']->get('train_station_lng') ?? 6.0501),
        ],
        'airport' => [
          'enabled' => (bool) ($data['config']->get('airport_enabled') ?? FALSE),
          'name' => $data['config']->get('airport_name') ?? 'Geneva Airport (GVA)',
          'description' => $data['config']->get('airport_description') ?? 'EMB - International airport with direct tram connection to CERN via line 18. Travel time: ~30 minutes.',
          'lat' => (float) ($data['config']->get('airport_lat') ?? 46.2381),
          'lng' => (float) ($data['config']->get('airport_lng') ?? 6.1089),
        ],
        'busStation' => [
          'enabled' => (bool) ($data['config']->get('bus_station_enabled') ?? FALSE),
        ],
        'defaultTransportMode' => $data['config']->get('default_transport_mode') ?? 'transit',
        'taxiService' => $data['config']->get('taxi_service') ?? '+41 22 331 41 33',
        'carRental' => $data['config']->get('car_rental') ?? 'Available at Geneva Airport and main train station',
      ],

      // Parking settings.
      'parking' => [
        'mainParking' => [
          'enabled' => (bool) ($data['config']->get('main_parking_enabled') ?? FALSE),
          'name' => $data['config']->get('main_parking_name') ?? 'CERN Visitor Parking',
          'description' => $data['config']->get('main_parking_description') ?? 'Visitor parking near the reception building. Limited spaces available.',
          'lat' => (float) ($data['config']->get('main_parking_lat') ?? 46.2335),
          'lng' => (float) ($data['config']->get('main_parking_lng') ?? 6.052),
        ],
        'capacity' => [
          'total' => (int) ($data['config']->get('parking_capacity_total') ?? 40),
          'disabled' => (int) ($data['config']->get('parking_capacity_disabled') ?? 6),
          'family' => (int) ($data['config']->get('parking_capacity_family') ?? 2),
          'ev' => (int) ($data['config']->get('parking_capacity_ev') ?? 4),
        ],
        'feeType' => $data['config']->get('parking_fee_type') ?? 'free',
        'hours' => $data['config']->get('parking_hours') ?? 'Mon-Fri: 8:00-18:00, Weekends: 9:00-17:00',
      ],

      // Routing & Integration.
      'routing' => [
        'service' => $data['config']->get('route_service') ?? 'simulated',
        'mapboxToken' => $data['config']->get('mapbox_token') ?? '',
        'googleApiKey' => $data['config']->get('google_api_key') ?? '',
      ],

      // Custom data.
      'nearbyMarkers' => $this->parseNearbyMarkers($data['config']->get() ?? []),
      'customMarkers' => $this->parseCustomMarkers($data['config']->get('custom_markers_json') ?? ''),
      'parkingZones' => $this->parseParkingZones($data['config']->get('parking_zones_json') ?? ''),

      // Advanced settings.
      'advanced' => [
        'enableDebug' => (bool) ($data['config']->get('enable_debug') ?? FALSE),
        'logLevel' => $data['config']->get('log_level') ?? 'error',
      ],

      // Country options.
      'countryOptions' => $this->getCountryOptionsArray(),

      // Map type options.
      'mapTypeOptions' => [
        'openstreetmap' => 'OpenStreetMap',
        'google' => 'Google Maps',
        'mapbox' => 'Mapbox',
        'cartodb' => 'CartoDB',
      ],

      // Route service options.
      'routeServiceOptions' => [
        'simulated' => 'Simulated Routes',
        'osrm' => 'OSRM (Open Source)',
        'mapbox' => 'Mapbox Directions',
        'google' => 'Google Directions API',
      ],

      // Transport mode options.
      'transportModeOptions' => [
        'driving' => 'Driving',
        'walking' => 'Walking',
        'cycling' => 'Cycling',
        'transit' => 'Public Transport',
      ],

      // Parking fee type options.
      'parkingFeeTypeOptions' => [
        'free' => 'Free Parking',
        'hourly' => 'Hourly Rate',
        'daily' => 'Daily Rate',
        'weekly' => 'Weekly Rate',
      ],

      // Address format options.
      'addressFormatOptions' => [
        'single_line' => 'Single line (Street, City, Country)',
        'multi_line' => 'Multi-line (US/International format)',
        'localized' => 'Localized format (based on country)',
      ],

      // Zoom control position options.
      'zoomControlPositionOptions' => [
        'topleft' => 'Top Left',
        'topright' => 'Top Right',
        'bottomleft' => 'Bottom Left',
        'bottomright' => 'Bottom Right',
      ],

      // Log level options.
      'logLevelOptions' => [
        'error' => 'Errors only',
        'warn' => 'Warnings and errors',
        'info' => 'All information',
      ],
    ];

    // Site marker configuration.
    $site_marker = [
      'coords' => [$center_lat, $center_lng],
      'title' => $data['site_config']->get('name'),
      'description' => $data['site_config']->get('slogan') ?: $data['site_config']->get('name'),
      'icon' => 'home',
      'color' => '#3388ff',
      'emoji' => '🏠',
    ];

    // Parse custom markers JSON.
    $custom_markers = [];
    $json_input = $data['config']->get('custom_markers_json');

    if (!empty(trim($json_input))) {
      $parsed_markers = json_decode($json_input, TRUE);
      if (isset($parsed_markers['markers']) && is_array($parsed_markers['markers'])) {
        $custom_markers = $parsed_markers['markers'];
      }
    }

    $module_path = $this->moduleExtensionList->getPath('map');

    // Get business name from site configuration.
    $business_name = $data['site_config']->get('name') ?: $this->t('Your Business');


    // Build render array.
    $build = [
      '#theme' => 'map_page',
      '#cache' => [
        'tags' => ['config:map.settings', 'config:system.site'],
        'contexts' => [],
      ],

      // Pass variables to template.
      '#business_name' => $business_name,
      '#address_city' => $data['config']->get('city') ?? '',
      '#default_address' => $this->formatDefaultAddress($data['config']),
      '#default_hours' => $this->t('Monday to Friday from 9:00 AM to 6:00 PM'),

    ];
    $build['#attached']['library'][] = 'map/map-page-4';
    $build['#attached']['drupalSettings']['map'] = [
      'config' => $map_config,
      'siteMarker' => $site_marker,
      'customMarkers' => $custom_markers,
      // 'test' => 123,
    ];

    if ($this->moduleExtensionList->get('business_identity')->status == 1) {


      // dump ($this->config('business_identity.settings')->getRawData());
      // exit;

      $build['#attached']['drupalSettings']['map']['business_identity_settings'] = $this->config('business_identity.settings')->getRawData();
    }

    // $build['#business_identity_settings'] = "QQQ";
    // dump ($build['#attached']['drupalSettings']['map']);
    // exit;

    return $build;
  }

  /**
   * Get country options as an array.
   *
   * @return array
   *   Array of country codes and names.
   */
  protected function getCountryOptionsArray() {
    return [
      'CH' => 'Switzerland',
      'IT' => 'Italy',
      'US' => 'United States',
      'GB' => 'United Kingdom',
      'DE' => 'Germany',
      'FR' => 'France',
      'ES' => 'Spain',
      'CA' => 'Canada',
      'AU' => 'Australia',
      'JP' => 'Japan',
      'CN' => 'China',
      'IN' => 'India',
      'BR' => 'Brazil',
      'MX' => 'Mexico',
      'RU' => 'Russia',
      'ZA' => 'South Africa',
    ];
  }

  /**
   * Parse custom markers JSON.
   *
   * @param string $json_input
   *   The JSON input string.
   *
   * @return array
   *   Array of parsed markers.
   */
  protected function parseCustomMarkers($json_input) {
    $markers = [];

    if (!empty(trim($json_input))) {
      $parsed = json_decode($json_input, TRUE);
      if (json_last_error() === JSON_ERROR_NONE) {
        // Support multiple formats.
        if (isset($parsed['markers']) && is_array($parsed['markers'])) {
          $markers = $parsed['markers'];
        }
        elseif (is_array($parsed)) {
          // Assume it's already an array of markers.
          $markers = $parsed;
        }
      }
    }

    return $markers;
  }

  /**
   * Parse parking zones JSON.
   *
   * @param string $json_input
   *   The JSON input string.
   *
   * @return array
   *   Array of parsed parking zones.
   */
  protected function parseParkingZones($json_input) {
    $zones = [];

    if (!empty(trim($json_input))) {
      $parsed = json_decode($json_input, TRUE);
      if (json_last_error() === JSON_ERROR_NONE && is_array($parsed)) {
        $zones = $parsed;
      }
    }

    return $zones;
  }

  /**
   * Format address for JavaScript from config object.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The config object.
   *
   * @return string
   *   Formatted address string.
   */
  private function formatAddressForConfig($config) {
    $parts = [];

    $street_address = $config->get('street_address');
    if (!empty($street_address)) {
      $parts[] = $street_address;
    }

    $street_address2 = $config->get('street_address2');
    if (!empty($street_address2)) {
      $parts[] = $street_address2;
    }

    $city = $config->get('city');
    if (!empty($city)) {
      $city_part = $city;
      $postal_code = $config->get('postal_code');
      if (!empty($postal_code)) {
        $city_part = $postal_code . ' ' . $city_part;
      }
      $state_province = $config->get('state_province');
      if (!empty($state_province)) {
        $city_part .= ', ' . $state_province;
      }
      $parts[] = $city_part;
    }

    $country = $config->get('country');
    if (!empty($country)) {
      // Get country name from options.
      $country_options = $this->getCountryOptionsArray();
      $country_name = $country_options[$country] ?? $country;
      $parts[] = $country_name;
    }

    return implode("\n", $parts);
  }


    /**
   * Format default address for template.
   *
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The config object.
   *
   * @return string
   *   Formatted address string.
   */
  private function formatDefaultAddress($config) {
    $street = $config->get('street_address') ?? '';
    $city = $config->get('city') ?? '';
    $postal_code = $config->get('postal_code') ?? '';

    if ($street && $city) {
      $address = $street . ' - ';
      if ($postal_code) {
        $address .= $postal_code . ' ';
      }
      $address .= $city;
      return $address;
    }

    return $this->t('Business Street, 123 - 00100 Rome');
  }










/**
 * Parse nearby markers from configured views.
 *
 * @param array $config
 *   Configuration array containing nearby interests settings.
 *
 * @return array
 *   Array of parsed markers with coordinates and information.
 */
// protected function parseNearbyMarkers(array $config) {
//   $markers = [];
//
//   // Check if nearby interests is enabled
//   if (empty($config['nearby_interests_enabled'])) {
//     return $markers;
//   }
//
//   // Check if a view is selected
//   $selected_view = !empty($config['nearby_interests_view'])
//     ? $config['nearby_interests_view']
//     : (isset($config['nearby_interests_views'][0]) ? $config['nearby_interests_views'][0] : '');
//
//   if (empty($selected_view)) {
//     return $markers;
//   }
//
//   try {
//     // Load the selected view
//     $view = Views::getView($selected_view);
//
//     if (!$view) {
//       \Drupal::logger('map')->error('View @view not found for nearby interests.', ['@view' => $selected_view]);
//       return $markers;
//     }
//
//     // Set display and arguments if needed
//     $view->setDisplay('default');
//
//     // Get the current map center coordinates for filtering by distance
//     $map_center_lat = $config['custom_lat'] ?? $this->getDefaultMapCenterLat();
//     $map_center_lng = $config['custom_lng'] ?? $this->getDefaultMapCenterLng();
//     $radius_km = !empty($config['nearby_radius']) ? (float) $config['nearby_radius'] : 50;
//     $limit = !empty($config['nearby_limit']) ? (int) $config['nearby_limit'] : 20;
//
//     // Execute the view
//     $view->execute();
//
//     $results = $view->result;
//     $count = 0;
//
//     foreach ($results as $index => $row) {
//       // Stop if we've reached the limit
//       if ($count >= $limit) {
//         break;
//       }
//
//       // Extract coordinates from the view result
//       // This assumes your view has fields named 'field_latitude' and 'field_longitude'
//       // Adjust these field names based on your actual view configuration
//       $entity = $row->_entity;
//
//       if (!$entity) {
//         continue;
//       }
//
//       // Try different common field names for latitude/longitude
//       $lat = $this->extractFieldValue($entity, ['field_latitude', 'field_lat', 'latitude', 'lat']);
//       $lng = $this->extractFieldValue($entity, ['field_longitude', 'field_lng', 'longitude', 'lng', 'lon']);
//
//       // If coordinates are not found, try to get from address field
//       if (empty($lat) || empty($lng)) {
//         $coordinates = $this->extractCoordinatesFromAddress($entity);
//         $lat = $coordinates['lat'] ?? null;
//         $lng = $coordinates['lng'] ?? null;
//       }
//
//
//       if (empty($lat) || empty($lng)) {
//         continue;
//       }
//
//       // Calculate distance from map center (optional filtering)
//       $distance = $this->calculateDistance($map_center_lat, $map_center_lng, $lat, $lng);
//
//       // Filter by radius if configured.
//       // @todo: too silent when pois are much away than radius_km. Add diagnostic.
//       // if ($radius_km > 0 && $distance > $radius_km) {
//       //   continue;
//       // }
//
//       // Extract title
//       $title = $this->extractFieldValue($entity, ['title', 'name', 'field_title', 'field_name', 'label']);
//
//       // Extract description/body
//       $description = $this->extractFieldValue($entity, [
//         'body', 'field_description', 'description',
//         'field_body', 'field_summary'
//       ]);
//
//       // Extract category/tags if grouping is enabled
//       $category = null;
//
//       // dump ($entity);
//
//       // if (!empty($config['nearby_categories'])) {
//         $category = $this->extractFieldValue($entity, [
//           'field_poi_type', 'field_category', 'field_tags', 'field_type',
//           'category', 'type', 'field_taxonomy'
//         ]);
//       // }
//
//       // Build marker data
//       $marker = [
//         'category' => $category,
//         'description' => $description ?: '',
//         'distance' => round($distance, 2),
//         'lat' => (float) $lat,
//         'lng' => (float) $lng,
//         'enabled' => 1,
//         'icon' => $this->getMarkerIcon($category),
//         'popup_enabled' => !empty($config['nearby_popup']),
//         'title' => $title ?: $this->t('Point of Interest'),
//         'type' => 'nearby_interest',
//       ];
//
//       // dump ($marker);
//
//       // Add entity information for potential linking
//       if ($entity->hasLinkTemplate('canonical')) {
//         try {
//           $marker['url'] = $entity->toUrl()->toString();
//         } catch (\Exception $e) {
//           // URL generation failed, skip
//         }
//       }
//
//       $markers[] = $marker;
//       $count++;
//     }
//
//     // Sort by distance if needed
//     if (!empty($markers)) {
//       usort($markers, function($a, $b) {
//         return $a['distance'] <=> $b['distance'];
//       });
//     }
//
//     // Apply clustering if enabled
//     if (!empty($config['nearby_cluster']) && count($markers) > 1) {
//       $markers = $this->applyClustering($markers);
//     }
//
//     \Drupal::logger('map')->info('Loaded @count nearby interest markers from view @view.', [
//       '@count' => count($markers),
//       '@view' => $selected_view,
//     ]);
//
//   } catch (\Exception $e) {
//     \Drupal::logger('map')->error('Error parsing nearby markers: @error', ['@error' => $e->getMessage()]);
//   }
//
//   return $markers;
// }



/**
 * Parse nearby markers from configured views.
 *
 * @param array $config
 *   Configuration array containing nearby interests settings.
 *
 * @return array
 *   Array of parsed markers with coordinates and information.
 */
protected function parseNearbyMarkers(array $config) {
  $markers = [];

  // Check if nearby interests is enabled
  if (empty($config['nearby_interests_enabled'])) {
    return $markers;
  }

  // Check if a view is selected
  $selected_view = !empty($config['nearby_interests_view'])
    ? $config['nearby_interests_view']
    : (isset($config['nearby_interests_views'][0]) ? $config['nearby_interests_views'][0] : '');

  if (empty($selected_view)) {
    return $markers;
  }

  try {
    // Load the selected view
    $view = Views::getView($selected_view);

    if (!$view) {
      \Drupal::logger('map')->error('View @view not found for nearby interests.', ['@view' => $selected_view]);
      return $markers;
    }

    // Set display and arguments if needed
    $view->setDisplay('default');

    // Get the current map center coordinates for filtering by distance
    $map_center_lat = $config['custom_lat'] ?? $this->getDefaultMapCenterLat();
    $map_center_lng = $config['custom_lng'] ?? $this->getDefaultMapCenterLng();
    $radius_km = !empty($config['nearby_radius']) ? (float) $config['nearby_radius'] : 50;
    $limit = !empty($config['nearby_limit']) ? (int) $config['nearby_limit'] : 20;

    // Execute the view
    $view->execute();

    $results = $view->result;
    $count = 0;

    // Get Geofield services (do this once to avoid repeated service calls)
    $geophp = \Drupal::service('geofield.geophp');

    foreach ($results as $index => $row) {
      // Stop if we've reached the limit
      if ($count >= $limit) {
        break;
      }

      // Extract coordinates from the view result
      $entity = $row->_entity;

      if (!$entity) {
        continue;
      }

      // Initialize coordinates
      $lat = null;
      $lng = null;

      // METHOD 1: Try Geofield first (check common geofield field names)
      $geofield_coordinates = $this->extractCoordinatesFromGeofield($entity, [
        'field_geofield', 'field_location', 'field_geo', 'field_coordinates',
        'field_point', 'field_map', 'geofield'
      ]);

      if (!empty($geofield_coordinates)) {
        $lat = $geofield_coordinates['lat'];
        $lng = $geofield_coordinates['lon'];
      }

      // If no geofield found, try regular latitude/longitude fields
      if (empty($lat) || empty($lng)) {
        $lat = $this->extractFieldValue($entity, ['field_latitude', 'field_lat', 'latitude', 'lat']);
        $lng = $this->extractFieldValue($entity, ['field_longitude', 'field_lng', 'longitude', 'lng', 'lon']);
      }

      // If coordinates are still not found, try to get from address field
      if (empty($lat) || empty($lng)) {
        $coordinates = $this->extractCoordinatesFromAddress($entity);
        $lat = $coordinates['lat'] ?? null;
        $lng = $coordinates['lng'] ?? null;
      }

      if (empty($lat) || empty($lng)) {
        continue;
      }

      // Calculate distance from map center (optional filtering)
      $distance = $this->calculateDistance($map_center_lat, $map_center_lng, $lat, $lng);

      // Filter by radius if configured
      if ($radius_km > 0 && $distance > $radius_km) {
        continue;
      }

      // Extract title
      $title = $this->extractFieldValue($entity, ['title', 'name', 'field_title', 'field_name', 'label']);

      // Extract description/body
      $description = $this->extractFieldValue($entity, [
        'body', 'field_description', 'description',
        'field_body', 'field_summary'
      ]);

      // Extract category/tags
      $category = $this->extractFieldValue($entity, [
        'field_poi_type', 'field_category', 'field_tags', 'field_type',
        'category', 'type', 'field_taxonomy'
      ]);

      // Build marker data
      $marker = [
        'category' => $category,
        'description' => $description ?: '',
        'distance' => round($distance, 2),
        'lat' => (float) $lat,
        'lng' => (float) $lng,
        'enabled' => 1,
        'icon' => $this->getMarkerIcon($category),
        'popup_enabled' => !empty($config['nearby_popup']),
        'title' => $title ?: $this->t('Point of Interest'),
        'type' => 'nearby_interest',
      ];

      // Add entity information for potential linking
      if ($entity->hasLinkTemplate('canonical')) {
        try {
          $marker['url'] = $entity->toUrl()->toString();
        } catch (\Exception $e) {
          // URL generation failed, skip
        }
      }

      $markers[] = $marker;
      $count++;
    }

    // Sort by distance if needed
    if (!empty($markers)) {
      usort($markers, function($a, $b) {
        return $a['distance'] <=> $b['distance'];
      });
    }

    // Apply clustering if enabled
    if (!empty($config['nearby_cluster']) && count($markers) > 1) {
      $markers = $this->applyClustering($markers);
    }

    \Drupal::logger('map')->info('Loaded @count nearby interest markers from view @view.', [
      '@count' => count($markers),
      '@view' => $selected_view,
    ]);

  } catch (\Exception $e) {
    \Drupal::logger('map')->error('Error parsing nearby markers: @error', ['@error' => $e->getMessage()]);
  }

  return $markers;
}

/**
 * Extract coordinates from Geofield using Method 1.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to extract coordinates from.
 * @param array $possible_field_names
 *   Array of possible geofield field names to check.
 *
 * @return array|null
 *   Array with 'lat' and 'lon' keys, or NULL if not found.
 */
protected function extractCoordinatesFromGeofield($entity, array $possible_field_names = []) {
  // Default field names to check
  $default_field_names = ['field_geofield', 'field_location', 'field_geo', 'field_coordinates'];
  $field_names = !empty($possible_field_names) ? $possible_field_names : $default_field_names;

  foreach ($field_names as $field_name) {
    if (!$entity->hasField($field_name) || $entity->get($field_name)->isEmpty()) {
      continue;
    }

    try {
      $geofield_item = $entity->get($field_name)->first();
      $wkt = $geofield_item->value;

      // Use Geofield's GeoPHP service to parse WKT
      $geometry = \Drupal::service('geofield.geophp')->load($wkt);

      if ($geometry && $geometry->geometryType() === 'Point') {
        return [
          'lat' => $geometry->y(), // Latitude is the Y coordinate
          'lon' => $geometry->x(), // Longitude is the X coordinate
        ];
      }

      // If it's not a Point, try other geometry types
      if ($geometry) {
        // For other geometry types (Polygon, LineString, etc.), get the centroid
        $centroid = $geometry->centroid();
        if ($centroid) {
          return [
            'lat' => $centroid->y(),
            'lon' => $centroid->x(),
          ];
        }
      }

    } catch (\Exception $e) {
      // Log but continue to try other fields
      \Drupal::logger('map')->debug('Error extracting from geofield @field: @error', [
        '@field' => $field_name,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  return null;
}

/**
 * Helper function to extract field value from entity.
 *
 * @param \Drupal\Core\Entity\EntityInterface $entity
 *   The entity to extract from.
 * @param array $field_names
 *   Array of possible field names to check.
 *
 * @return mixed|null
 *   The field value or NULL if not found.
 */
protected function extractFieldValue($entity, array $field_names) {
  foreach ($field_names as $field_name) {
    if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
      $field = $entity->get($field_name);

      // Handle different field types
      if ($field->getFieldDefinition()->getType() === 'entity_reference') {
        // For entity reference fields, get the referenced entity label
        $referenced_entity = $field->entity;
        if ($referenced_entity) {
          return $referenced_entity->label();
        }
      } elseif ($field->getFieldDefinition()->getType() === 'text_with_summary' ||
                $field->getFieldDefinition()->getType() === 'text_long') {
        // For text fields with summary, get the value
        $value = $field->value;
        if (!empty($value)) {
          // Strip HTML tags for cleaner output
          return strip_tags($value);
        }
      } else {
        // For simple fields
        return $field->value;
      }
    }
  }

  return null;
}









/**
 * Helper method to extract field value from entity.
 */
// protected function extractFieldValue($entity, array $field_names) {
//   foreach ($field_names as $field_name) {
//     if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
//
//       $first_term = $this->getFirstReferencedTerm($entity, $field_name);
//
//       if ($first_term) {
//         return $first_term['name'];
//       }
//
//       $value = $entity->get($field_name)->value;
//       if (!empty($value)) {
//         return $value;
//       }
//     }
//   }
//   return null;
// }



public function getFirstReferencedTerm($entity, $field_name) {
  if (!$entity->hasField($field_name)) {
    return NULL;
  }

  $field = $entity->get($field_name);

  // Check if field has values and get the first referenced entity
  if (!$field->isEmpty() && $first_entity = $field->entity) {
    // Check if it's a taxonomy term
    if ($first_entity instanceof \Drupal\taxonomy\TermInterface) {
      return [
        'id' => $first_entity->id(),
        'name' => $first_entity->label(),
        'entity' => $first_entity
      ];
    }
  }

  return NULL;
}





/**
 * Extract coordinates from address field.
 */
/**
 * Extract coordinates from address or geolocation field.
 */
protected function extractCoordinatesFromAddress($entity) {
  $coordinates = ['lat' => null, 'lng' => null];

  // Try common geolocation/address field names
  $field_names = [
    'field_geolocation', // Common geolocation field name
    'field_location',    // Common location field name
    'geolocation',       // Simpler field name
    'location',          // Simple field name
    'field_address',     // Address field (might have geocoding)
    'address',           // Simple address field
  ];

  foreach ($field_names as $field_name) {
    if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
      $field = $entity->get($field_name)->first();

      // Check if this is a geolocation field
      if ($field instanceof \Drupal\geolocation\Plugin\Field\FieldType\GeolocationItem) {
        // For geolocation fields, get lat/lng directly
        $coordinates['lat'] = $field->get('lat')->getValue();
        $coordinates['lng'] = $field->get('lng')->getValue();
        break;
      }
      // Check if it's a standard address field with coordinates
      elseif (method_exists($field, 'get') && $field->get('lat')) {
        $coordinates['lat'] = $field->get('lat')->getValue();
        $coordinates['lng'] = $field->get('lng')->getValue();
        break;
      }
      // Check for latitude/longitude properties
      elseif (property_exists($field, 'latitude') && property_exists($field, 'longitude')) {
        $coordinates['lat'] = $field->latitude;
        $coordinates['lng'] = $field->longitude;
        break;
      }
      // Check for lat/lng properties directly
      elseif (property_exists($field, 'lat') && property_exists($field, 'lng')) {
        $coordinates['lat'] = $field->lat;
        $coordinates['lng'] = $field->lng;
        break;
      }
    }
  }

  // If still no coordinates, try to get from field properties directly
  if (empty($coordinates['lat']) || empty($coordinates['lng'])) {
    foreach ($field_names as $field_name) {
      if ($entity->hasField($field_name) && !$entity->get($field_name)->isEmpty()) {
        $field_value = $entity->get($field_name)->getValue();
        if (!empty($field_value[0])) {
          $field_data = $field_value[0];

          // Check common geolocation data structures
          if (isset($field_data['lat']) && isset($field_data['lng'])) {
            $coordinates['lat'] = $field_data['lat'];
            $coordinates['lng'] = $field_data['lng'];
            break;
          } elseif (isset($field_data['latitude']) && isset($field_data['longitude'])) {
            $coordinates['lat'] = $field_data['latitude'];
            $coordinates['lng'] = $field_data['longitude'];
            break;
          } elseif (isset($field_data['value']) && is_array($field_data['value'])) {
            // Some fields store coordinates in a 'value' array
            if (isset($field_data['value']['lat']) && isset($field_data['value']['lng'])) {
              $coordinates['lat'] = $field_data['value']['lat'];
              $coordinates['lng'] = $field_data['value']['lng'];
              break;
            }
          }
        }
      }
    }
  }

  return $coordinates;
}





/**
 * Calculate distance between two points in kilometers.
 */
protected function calculateDistance($lat1, $lon1, $lat2, $lon2) {
  $earth_radius = 6371; // Earth's radius in kilometers

  $dLat = deg2rad($lat2 - $lat1);
  $dLon = deg2rad($lon2 - $lon1);

  $a = sin($dLat/2) * sin($dLat/2) +
       cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
       sin($dLon/2) * sin($dLon/2);

  $c = 2 * atan2(sqrt($a), sqrt(1-$a));

  return $earth_radius * $c;
}

/**
 * Get appropriate marker icon based on category.
 */
protected function getMarkerIcon($category = null) {
  $icons = [
    'default' => 'leaflet-default-marker',
    'restaurant' => 'leaflet-restaurant-marker',
    'hotel' => 'leaflet-hotel-marker',
    'museum' => 'leaflet-museum-marker',
    'park' => 'leaflet-park-marker',
    'shop' => 'leaflet-shop-marker',
  ];

  if ($category && isset($icons[strtolower($category)])) {
    return $icons[strtolower($category)];
  }

  return $icons['default'];
}

/**
 * Apply simple clustering to markers.
 */
protected function applyClustering(array $markers, $cluster_distance = 0.01) {
  $clusters = [];
  $used = [];

  foreach ($markers as $i => $marker) {
    if (in_array($i, $used)) {
      continue;
    }

    $cluster = [$marker];
    $used[] = $i;

    foreach ($markers as $j => $other_marker) {
      if ($i === $j || in_array($j, $used)) {
        continue;
      }

      $distance = $this->calculateDistance(
        $marker['lat'], $marker['lng'],
        $other_marker['lat'], $other_marker['lng']
      );

      // Convert km to degrees (rough approximation)
      $distance_degrees = $distance / 111;

      if ($distance_degrees < $cluster_distance) {
        $cluster[] = $other_marker;
        $used[] = $j;
      }
    }

    if (count($cluster) === 1) {
      $clusters[] = $marker;
    } else {
      // Calculate cluster center
      $avg_lat = array_sum(array_column($cluster, 'lat')) / count($cluster);
      $avg_lng = array_sum(array_column($cluster, 'lng')) / count($cluster);

      $clusters[] = [
        'lat' => $avg_lat,
        'lng' => $avg_lng,
        'title' => $this->t('Cluster (@count items)', ['@count' => count($cluster)]),
        'description' => $this->formatClusterDescription($cluster),
        'type' => 'cluster',
        'cluster_items' => $cluster,
        'count' => count($cluster),
        'icon' => 'leaflet-cluster-marker',
        'popup_enabled' => true,
      ];
    }
  }

  return $clusters;
}

/**
 * Format cluster description for popup.
 */
protected function formatClusterDescription(array $cluster_items) {
  $items = [];
  foreach ($cluster_items as $item) {
    if (isset($item['title'])) {
      $items[] = $item['title'];
    }
  }

  return '<ul><li>' . implode('</li><li>', array_slice($items, 0, 5)) . '</li></ul>' .
         (count($items) > 5 ? '<p>' . $this->t('And @more more...', ['@more' => count($items) - 5]) . '</p>' : '');
}

/**
 * Get default map center latitude.
 */
protected function getDefaultMapCenterLat() {
  $config = $this->config('map.settings');
  return $config->get('custom_lat') ?: 41.9028;
}

/**
 * Get default map center longitude.
 */
protected function getDefaultMapCenterLng() {
  $config = $this->config('map.settings');
  return $config->get('custom_lng') ?: 12.4964;
}


























}
