<?php

namespace Drupal\map\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure map marker settings.
 */
class MapMarkerSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['map.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'map_marker_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('map.settings');

    $form['custom_markers_json'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Markers JSON'),
      '#default_value' => $config->get('custom_markers_json'),
      '#description' => $this->t('Enter markers in JSON format. Example:<br>
<pre>
{
  "markers": [
    {
      "coords": [42.406246, 12.686570],
      "zoom": 18,
      "title": "Il tuo B&B",
      "description": "Il tuo punto di partenza&lt;br&gt;&lt;br&gt;📍 Via del B&B, Cottanello&lt;br&gt;☎️ +39 123 456 789&lt;br&gt;📧 info@bbcottanello.it",
      "icon": "home",
      "color": "#4CAF50",
      "emoji": "🏠",
      "image": "https://images.unsplash.com/photo-1566073771259-6a8506099945"
    }
  ]
}</pre>'),
      '#rows' => 15,
      '#attributes' => [
        'style' => 'font-family: monospace; font-size: 12px;',
      ],
    ];

    $form['json_validation'] = [
      '#type' => 'markup',
      '#markup' => '<div id="json-validation-message"></div>',
    ];

    // Add validation via AJAX or JavaScript
    $form['#attached']['library'][] = 'map/map-admin';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $json_input = $form_state->getValue('custom_markers_json');

    // Only validate if JSON is provided
    if (!empty(trim($json_input))) {
      json_decode($json_input);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('custom_markers_json',
          $this->t('Invalid JSON format: %error', ['%error' => json_last_error_msg()])
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('map.settings');

    $config->set('custom_markers_json', $form_state->getValue('custom_markers_json'))
           ->save();

    parent::submitForm($form, $form_state);
  }

}
