<?php

declare(strict_types=1);

namespace Drupal\mapsemble;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Link;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of maps.
 */
final class MapsembleMapListBuilder extends ConfigEntityListBuilder {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  public ConfigFactoryInterface $configFactory;

  /**
   * Constructs a new MapsembleMapListBuilder object.
   *
   *   The entity type.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($entity_type, $storage);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(
    ContainerInterface $container,
    EntityTypeInterface $entity_type,
  ) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('config.factory')
    );
  }

  /**
   * Renders the list of maps.
   *
   * @return array
   *   A render array.
   */
  public function render() {
    if (!$this->configFactory->get('mapsemble.settings')->get('client_id')) {
      $credentials_link = Link::fromTextAndUrl(
        new TranslatableMarkup('here'),
        Url::fromRoute('mapsemble.settings')
      );
      $createlink = Link::fromTextAndUrl(
        new TranslatableMarkup('create a map'),
        Url::fromUri(
          (Settings::get(
            'mapsemble_app_base'
          ) ?? MapsembleApi::MAPSEMBLE_APP_BASE) . '/integrations/drupal',
          [
            'attributes' => [
              'target' => '_blank',
            ],
          ]
        )
      );
      $this->messenger()->addWarning(
        $this->t(
          'Please @create_link in Mapsemble first. Also configure your API credentials @credentials_link.',
          [
            '@create_link' => $createlink->toString(),
            '@credentials_link' => $credentials_link->toString(),
          ]
        )
      );
    }

    return parent::render();
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['label'] = $this->t('Label');
    $header['id'] = $this->t('Machine name');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    /** @var \Drupal\mapsemble\MapsembleMapInterface $entity */
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(EntityInterface $entity) {
    $operations = parent::getOperations($entity);

    // Add the "Sync All Entities" operation.
    $operations['sync_all_entities'] = [
      'title' => $this->t('Sync entities'),
      'url' => Url::fromRoute('mapsemble.sync_all_entities', [
        'mapsemble_map' => $entity->id(),
      ]),
      'weight' => 10,
    ];

    return $operations;
  }

}
