<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Plugin\Block;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mapsemble\Entity\MapsembleMap;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a mapsemble map block.
 */
#[Block(
  id: 'mapsemble_map',
  admin_label: new TranslatableMarkup('Mapsemble map'),
  category: new TranslatableMarkup('Mapsemble'),
)]
final class MapBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new MapBlock.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'map' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form['map'] = [
      '#type' => 'select',
      '#title' => $this->t('Map'),
      '#default_value' => $this->configuration['map'],
      '#required' => TRUE,
      '#options' => array_map(
        fn(MapsembleMap $map) => $map->label(),
        $this->entityTypeManager->getStorage('mapsemble_map')->loadMultiple()
      ),
    ];
    $form['height'] = [
      '#type' => 'textfield',
      '#size' => 10,
      '#title' => $this->t('Height'),
      '#default_value' => $this->configuration['height'] ?? '400px',
      '#description' => $this->t('Height of the map using any valid CSS value.'),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    $this->configuration['map'] = $form_state->getValue('map');
    $this->configuration['height'] = $form_state->getValue('height');
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $mapId = $this->configuration['map'];
    if (!$mapId) {
      return [];
    }
    /** @var \Drupal\mapsemble\Entity\MapsembleMap $map */
    $map = $this->entityTypeManager->getStorage('mapsemble_map')->load($mapId);

    if (!$map) {
      return [];
    }
    $build['map'] = [
      '#type' => 'mapsemble_map',
      '#map' => $map,
      '#height' => $this->configuration['height'],
    ];

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  protected function blockAccess(AccountInterface $account): AccessResult {
    return AccessResult::allowedIf('access mapsemble');
  }

}
