<?php

namespace Drupal\mapsemble\Controller;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Site\Settings;
use Drupal\Core\Theme\ThemeManager;
use Drupal\language\Plugin\LanguageNegotiation\LanguageNegotiationUrl;
use Drupal\mapsemble\Entity\MapsembleMap;
use Drupal\mapsemble\MapsembleApi;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Render\Renderer;
use Drupal\Core\Asset\AssetResolver;
use Drupal\Core\Asset\AttachedAssets;
use Drupal\Core\Asset\LibraryDependencyResolverInterface;
use Drupal\Core\Asset\AssetCollectionRendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for bare content without site theming.
 */
abstract class ContentControllerBase extends MapsembleControllerBase {

  /**
   * Constructs a new MapsembleContentControllerBase object.
   *
   * @param \Drupal\Core\Render\Renderer $renderer
   *   The renderer service.
   * @param \Drupal\Core\Asset\AssetResolver $assetResolver
   *   The asset resolver.
   * @param \Drupal\Core\Asset\AssetCollectionRendererInterface $cssCollectionRenderer
   *   The CSS collection renderer.
   * @param \Drupal\Core\Asset\AssetCollectionRendererInterface $jsCollectionRenderer
   *   The JS collection renderer.
   * @param \Drupal\Core\Asset\LibraryDependencyResolverInterface $libraryDependencyResolver
   *   The library dependency resolver.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Theme\ThemeManager $themeManager
   *   The theme manager.
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher service.
   * @param \Symfony\Component\HttpFoundation\Request $currentRequest
   *   The current request.
   */
  public function __construct(
    protected readonly Renderer $renderer,
    protected readonly AssetResolver $assetResolver,
    protected readonly AssetCollectionRendererInterface $cssCollectionRenderer,
    protected readonly AssetCollectionRendererInterface $jsCollectionRenderer,
    protected readonly LibraryDependencyResolverInterface $libraryDependencyResolver,
    protected readonly RequestStack $requestStack,
    EntityTypeManagerInterface $entityTypeManager,
    protected readonly ThemeManager $themeManager,
    protected readonly EventDispatcherInterface $eventDispatcher,
    protected readonly Request $currentRequest,
  ) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('asset.resolver'),
      $container->get('asset.css.collection_renderer'),
      $container->get('asset.js.collection_renderer'),
      $container->get('library.dependency_resolver'),
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('theme.manager'),
      $container->get('event_dispatcher'),
      $container->get('request_stack')->getCurrentRequest(),
    );
  }

  /**
   * Returns the base key used to resolve view mode setting and templates.
   *
   * @return string
   *   The base key, e.g., 'card' or 'popup'.
   */
  abstract public function getBase(): string;

  /**
   * Returns bare content without site theme.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response object.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function __invoke() {
    $this->handleRequest();

    /** @var \Drupal\mapsemble\Entity\MapsembleMap[] $mapsembleMaps */
    $mapsembleMaps = $this->entityTypeManager->getStorage('mapsemble_map')
      ->loadMultiple();
    $mapsembleMap = NULL;
    foreach ($mapsembleMaps as $mapsembleMap) {
      if ($mapsembleMap->getMapId() == $this->mapId) {
        break;
      }
    }
    if (!$mapsembleMap) {
      throw new BadRequestException(
        'No Mapsemble Map found in Drupal configuration.'
      );
    }

    assert($mapsembleMap instanceof MapsembleMap);

    $entity_type_id = $mapsembleMap->getSettings()['entity_type_id'];
    $entity = $this->entityTypeManager()->getStorage($entity_type_id)->load(
      $this->remoteId
    );
    $view = $this->entityTypeManager->getViewBuilder($entity_type_id)->view(
      $entity,
      $mapsembleMap->getSettings()[$this->getBase() . '_view_mode'],
      $this->language
    );

    $build['content'] = $view;

    $build['content']['#cache']['contexts'][] = 'url.query_args';
    $build['content']['#attached'] = [
      'library' => [
        'mapsemble/mapsemble',
      ],
    ];

    $active_theme = $this->themeManager->getActiveTheme();
    $libraries = $active_theme->getLibraries();

    foreach ($libraries as $library) {
      $build['#attached']['library'][] = $library;
    }

    $context = new RenderContext();
    $content = $this->renderer->executeInRenderContext(
      $context,
      function () use (&$build) {
        return $this->renderer->render($build, TRUE);
      }
    );
    $cacheable_metadata = $context->isEmpty() ? new CacheableMetadata(
    ) : $context->pop();

    $attachedAssets = AttachedAssets::createFromRenderArray($build);

    $css_assets = $this->assetResolver->getCssAssets($attachedAssets, FALSE);
    [
      $js_assets_header,
      $js_assets_footer,
    ] = $this->assetResolver->getJsAssets($attachedAssets, FALSE);

    $cssBuild = $this->cssCollectionRenderer->render($css_assets);
    $jsHeaderBuild = $this->jsCollectionRenderer->render($js_assets_header);
    $jsFooterBuild = $this->jsCollectionRenderer->render($js_assets_footer);

    $css = $this->renderer->renderInIsolation($cssBuild);
    $js_header = $this->renderer->renderInIsolation($jsHeaderBuild);
    $js_footer = $this->renderer->renderInIsolation($jsFooterBuild);

    // Build a minimal HTML structure with the assets.
    $html = <<<HTML
<!DOCTYPE html>
<html lang="$this->language">
  <head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    $css
    $js_header
  </head>
  <body style="overflow: hidden; max-width: 100%">
    $content
    $js_footer
  </body>
</html>
HTML;

    $domain = Settings::get(
      'mapsemble_app_base'
    ) ?? MapsembleApi::MAPSEMBLE_APP_BASE;

    // Get allowed domains for Content-Security-Policy.
    $allowed_domains = [$domain, MapsembleApi::MAPSEMBLE_BASE];

    // Check if domain-based language negotiation is enabled.
    $language_config = $this->config('language.negotiation');
    $url_config = $language_config->get('url');

    // If domain-based language negotiation is enabled, add language domains to allowed domains.
    if (isset($url_config['source']) && $url_config['source'] == LanguageNegotiationUrl::CONFIG_DOMAIN) {
      if (isset($url_config['domains']) && is_array($url_config['domains'])) {
        foreach ($url_config['domains'] as $lang_domain) {
          if (!empty($lang_domain)) {
            $allowed_domains[] = 'https://' . $lang_domain;
          }
        }
      }
    }
    else {
      $allowed_domains[] = $this->currentRequest->getSchemeAndHttpHost();
    }

    // Create a custom response.
    $response = new CacheableResponse();
    $response->setContent($html);
    $response->addCacheableDependency($cacheable_metadata);
    $response->headers->set(
      'Content-Security-Policy',
      'frame-ancestors ' . implode(' ', $allowed_domains)
    );

    $response->headers->set(
      'Access-Control-Allow-Origin',
      Settings::get(
        'mapsemble_app_base'
      ) ?? MapsembleApi::MAPSEMBLE_APP_BASE
    );

    return $response;
  }

}
