<?php

namespace Drupal\mapsemble\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\mapsemble\Entity\MapsembleMap;
use Drupal\mapsemble\Event\MapsembleFilterEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Mapsemble batch operations.
 */
class FilterController extends ControllerBase {

  /**
   * Constructs a FilterController object.
   *
   * @param \Symfony\Component\EventDispatcher\EventDispatcherInterface $eventDispatcher
   *   The event dispatcher.
   */
  public function __construct(
    private readonly EventDispatcherInterface $eventDispatcher,
  ) {

  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('event_dispatcher')
    );
  }

  /**
   * Extracts and validates the filter and map from the request.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   *
   * @return array
   *   An array containing the MapsembleMap entity and the filter ID: [MapsembleMap $map, string $filter].
   *
   * @throws \Symfony\Component\HttpFoundation\Exception\BadRequestException
   *   Thrown when required parameters are missing or invalid.
   */
  private function getMapAndFilterFromRequest(Request $request): array {
    $filter = $request->query->get('filter');
    if (!$filter) {
      throw new BadRequestException('No filter provided');
    }
    $mapId = $request->query->get('map');
    if (!$mapId) {
      throw new BadRequestException('No map provided');
    }

    // Load the map entity.
    $mapsembleMaps = MapsembleMap::loadMultiple();
    $map = NULL;
    foreach ($mapsembleMaps as $mapsembleMap) {
      $settings = $mapsembleMap->getSettings();
      if (isset($settings['mapsemble']['mapId']) && $settings['mapsemble']['mapId'] == $mapId) {
        $map = $mapsembleMap;
        break;
      }
    }
    if (!$map) {
      throw new BadRequestException('Map not found');
    }

    return [$map, $filter];
  }

  public function filter(Request $request): JsonResponse {
    $value = $request->query->all('value');
    if (!empty($value)) {
      if (is_array($value) && isset($value['value'])) {
        $value = $value['value'];
      }
    }

    [$map, $filter] = $this->getMapAndFilterFromRequest($request);

    // Create and dispatch the filter event.
    $event = new MapsembleFilterEvent($map, $filter, $value);
    $event = $this->eventDispatcher->dispatch($event, MapsembleFilterEvent::EVENT_NAME);

    // Get entity IDs from the event.
    $entityIds = $event->getEntityIds();

    // If no filtering was applied, return empty array.
    if ($entityIds === NULL) {
      $entityIds = [];
    }

    return new JsonResponse($entityIds);
  }

  public function filterOptions(Request $request): JsonResponse {
    $search = $request->query->get('search') ?? '';

    [$map, $filter] = $this->getMapAndFilterFromRequest($request);

    // Get the filter mappings from the map settings.
    $filterMappings = $map->get('settings')['filter_mappings'] ?? [];
    if (!isset($filterMappings[$filter])) {
      throw new BadRequestException('Filter not mapped');
    }

    $fieldName = $filterMappings[$filter];
    if (empty($fieldName)) {
      throw new BadRequestException('Filter not mapped to a field');
    }

    // Get entity type and bundle from map settings.
    $entityTypeId = $map->get('settings')['entity_type_id'] ?? '';
    $bundle = $map->get('settings')['bundle'] ?? '';
    if (!$entityTypeId || !$bundle) {
      throw new BadRequestException('Map not properly configured');
    }

    // Get the field definition.
    $bundleFields = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions($entityTypeId, $bundle);

    if (!isset($bundleFields[$fieldName])) {
      throw new BadRequestException('Field not found');
    }

    $fieldDefinition = $bundleFields[$fieldName];
    $fieldType = $fieldDefinition->getType();

    $options = [];

    // Handle different field types.
    if (in_array($fieldType, ['string', 'text'])) {
      // Text field: query for unique values.
      $options = $this->getUniqueTextFieldValues(
        $entityTypeId,
        $bundle,
        $fieldName,
        $search
      );
    }
    elseif (in_array($fieldType, ['list_string', 'list_integer', 'list_float']
    )) {
      // Options field: get from field definition.
      $options = $this->getOptionsFromFieldDefinition(
        $fieldDefinition,
        $search
      );
    }
    elseif ($fieldType === 'entity_reference') {
      // Check if it's a taxonomy reference.
      $settings = $fieldDefinition->getSettings();
      if (isset($settings['target_type']) && $settings['target_type'] === 'taxonomy_term') {
        // Taxonomy field: get from vocabulary.
        $options = $this->getTaxonomyOptions(
          $settings,
          $entityTypeId,
          $bundle,
          $fieldName,
          $search
        );
      }
      else {
        throw new BadRequestException(
          'Only taxonomy entity references are supported'
        );
      }
    }
    else {
      throw new BadRequestException('Unsupported field type');
    }

    return new JsonResponse($options);
  }

  /**
   * Get unique values for a text field.
   *
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   * @param string $fieldName
   *   The field name.
   * @param string $search
   *   The search string to filter by.
   *
   * @return array
   *   An associative array of unique values (value => value).
   */
  protected function getUniqueTextFieldValues(
    string $entityTypeId,
    string $bundle,
    string $fieldName,
    string $search = '',
  ): array {
    $entityStorage = $this->entityTypeManager()->getStorage($entityTypeId);
    $definition = $this->entityTypeManager()->getDefinition($entityTypeId);

    // Build base query.
    $query = $entityStorage->getQuery()
      ->condition('status', 1)
      ->accessCheck(FALSE);

    if ($definition->hasKey('bundle')) {
      $query->condition($definition->getKey('bundle'), $bundle);
    }

    // Only get entities that have a value for this field.
    $query->exists($fieldName);

    // Filter by search string if provided.
    if (!empty($search)) {
      $query->condition($fieldName, $search, 'CONTAINS');
    }

    $entityIds = $query->execute();

    $valueCounts = [];
    if ($entityIds) {
      $entities = $entityStorage->loadMultiple($entityIds);
      foreach ($entities as $entity) {
        if ($entity->hasField($fieldName) && !$entity->get($fieldName)->isEmpty(
          )) {
          $value = $entity->get($fieldName)->value;
          if (!empty($value)) {
            // Count occurrences of each value.
            if (!isset($valueCounts[$value])) {
              $valueCounts[$value] = 0;
            }
            $valueCounts[$value]++;
          }
        }
      }
    }

    // Sort by count (descending).
    arsort($valueCounts);

    // Limit to 10 results and convert to value => value format.
    $options = [];
    $count = 0;
    foreach ($valueCounts as $value => $occurrences) {
      if ($count >= 10) {
        break;
      }
      $options[$value] = $value;
      $count++;
    }

    return $options;
  }

  /**
   * Get options from a list field definition.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $fieldDefinition
   *   The field definition.
   * @param string $search
   *   The search string to filter by.
   *
   * @return array
   *   An associative array of options (value => label).
   */
  protected function getOptionsFromFieldDefinition(
    $fieldDefinition,
    string $search = '',
  ): array {
    // Get the allowed values from the field storage definition.
    $fieldStorage = $fieldDefinition->getFieldStorageDefinition();
    $allowedValues = [];
    if (method_exists($fieldStorage, 'getSetting')) {
      $allowedValues = $fieldStorage->getSetting('allowed_values');
      if (!is_array($allowedValues)) {
        $allowedValues = [];
      }
    }

    if (empty($allowedValues)) {
      return [];
    }

    // Filter allowed values by search string if provided.
    if (!empty($search)) {
      $filteredAllowedValues = [];
      foreach ($allowedValues as $key => $label) {
        if (stripos($label, $search) !== FALSE) {
          $filteredAllowedValues[$key] = $label;
        }
      }
      $allowedValues = $filteredAllowedValues;
    }

    if (empty($allowedValues)) {
      return [];
    }

    // Get entity type and bundle from the field definition.
    $entityTypeId = $fieldDefinition->getTargetEntityTypeId();
    $bundle = $fieldDefinition->getTargetBundle();
    $fieldName = $fieldDefinition->getName();

    // Count usage of each option value.
    $entityStorage = $this->entityTypeManager()->getStorage($entityTypeId);
    $definition = $this->entityTypeManager()->getDefinition($entityTypeId);

    $query = $entityStorage->getQuery()
      ->condition('status', 1)
      ->accessCheck(FALSE);

    if ($definition->hasKey('bundle') && $bundle) {
      $query->condition($definition->getKey('bundle'), $bundle);
    }

    $query->exists($fieldName);

    // Filter by search string using entity query if provided.
    if (!empty($search)) {
      $query->condition($fieldName, array_keys($allowedValues), 'IN');
    }

    $entityIds = $query->execute();

    $valueCounts = [];
    if ($entityIds) {
      $entities = $entityStorage->loadMultiple($entityIds);
      foreach ($entities as $entity) {
        if ($entity->hasField($fieldName) && !$entity->get($fieldName)->isEmpty(
          )) {
          $value = $entity->get($fieldName)->value;
          if (isset($allowedValues[$value])) {
            if (!isset($valueCounts[$value])) {
              $valueCounts[$value] = 0;
            }
            $valueCounts[$value]++;
          }
        }
      }
    }

    // Sort by count (descending).
    arsort($valueCounts);

    // Limit to 10 results and use labels from allowed values.
    $options = [];
    $count = 0;
    foreach ($valueCounts as $value => $occurrences) {
      if ($count >= 10) {
        break;
      }
      $options[$value] = $allowedValues[$value];
      $count++;
    }

    return $options;
  }

  /**
   * Get options from a taxonomy vocabulary.
   *
   * @param array $settings
   *   The field settings.
   * @param string $entityTypeId
   *   The entity type ID.
   * @param string $bundle
   *   The bundle.
   * @param string $fieldName
   *   The field name.
   * @param string $search
   *   The search string to filter by.
   *
   * @return array
   *   An associative array of options (term_id => term_name).
   */
  protected function getTaxonomyOptions(
    array $settings,
    string $entityTypeId = '',
    string $bundle = '',
    string $fieldName = '',
    string $search = '',
  ): array {
    // Get the target bundles (vocabularies).
    $vocabularies = $settings['handler_settings']['target_bundles'] ?? [];

    if (empty($vocabularies)) {
      return [];
    }

    // Load all terms from the vocabularies to get term names.
    $termStorage = $this->entityTypeManager()->getStorage('taxonomy_term');
    $allTerms = [];
    foreach ($vocabularies as $vocabulary) {
      $properties = ['vid' => $vocabulary];
      // Filter by search string if provided.
      if (!empty($search)) {
        // Load terms using entity query to filter by name.
        $termQuery = $termStorage->getQuery()
          ->condition('vid', $vocabulary)
          ->condition('name', $search, 'CONTAINS')
          ->accessCheck(FALSE);
        $termIds = $termQuery->execute();
        if ($termIds) {
          $terms = $termStorage->loadMultiple($termIds);
          foreach ($terms as $term) {
            $allTerms[$term->id()] = $term->getName();
          }
        }
      }
      else {
        $terms = $termStorage->loadByProperties($properties);
        foreach ($terms as $term) {
          $allTerms[$term->id()] = $term->getName();
        }
      }
    }

    if (empty($allTerms)) {
      return [];
    }

    // Count usage of each term in actual entities.
    $termCounts = [];
    if ($entityTypeId && $bundle && $fieldName) {
      $entityStorage = $this->entityTypeManager()->getStorage($entityTypeId);
      $definition = $this->entityTypeManager()->getDefinition($entityTypeId);

      $query = $entityStorage->getQuery()
        ->condition('status', 1)
        ->accessCheck(FALSE);

      if ($definition->hasKey('bundle')) {
        $query->condition($definition->getKey('bundle'), $bundle);
      }

      $query->exists($fieldName);

      // Filter by the term IDs that match the search.
      if (!empty($search)) {
        $query->condition($fieldName, array_keys($allTerms), 'IN');
      }

      $entityIds = $query->execute();

      if ($entityIds) {
        $entities = $entityStorage->loadMultiple($entityIds);
        foreach ($entities as $entity) {
          if ($entity->hasField($fieldName) && !$entity->get($fieldName)
            ->isEmpty()) {
            foreach ($entity->get($fieldName) as $item) {
              $termId = $item->target_id;
              if (isset($allTerms[$termId])) {
                if (!isset($termCounts[$termId])) {
                  $termCounts[$termId] = 0;
                }
                $termCounts[$termId]++;
              }
            }
          }
        }
      }
    }

    // Sort by count (descending).
    arsort($termCounts);

    // Limit to 10 results.
    $options = [];
    $count = 0;
    foreach ($termCounts as $termId => $occurrences) {
      if ($count >= 10) {
        break;
      }
      $options[$termId] = $allTerms[$termId];
      $count++;
    }

    return $options;
  }

}
