<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Element;

use Drupal\Core\Render\Attribute\RenderElement;
use Drupal\Core\Render\Element\RenderElementBase;
use Drupal\Core\Site\Settings;
use Drupal\mapsemble\Entity\MapsembleMap as MapEntity;
use Drupal\mapsemble\MapsembleApi;

/**
 * Provides a render element to display a mapsemble map.
 */
#[RenderElement(
  id: 'mapsemble_map',
)]
final class MapsembleMap extends RenderElementBase {

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    return [
      '#pre_render' => [
        [self::class, 'preRenderEntityElement'],
      ],
      '#map' => NULL,
      '#height' => '50vh',
    ];
  }

  /**
   * Mapsemble map element pre render callback.
   *
   * @param array $element
   *   An associative array containing the properties of the mapsemble_map
   *   element.
   *
   * @return array
   *   The modified element.
   */
  public static function preRenderEntityElement(array $element): array {
    $map = $element['#map'] ?? NULL;
    assert($map instanceof MapEntity);
    $element['widget'] = [
      '#type' => 'html_tag',
      '#tag' => 'script',
      '#value' => '',
      '#attributes' => [
        'src' => (Settings::get(
            'mapsemble_app_base'
          ) ?? MapsembleApi::MAPSEMBLE_APP_BASE) . '/widget.js',
      ],
    ];
    $element['container'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => '',
      '#attributes' => [
        'id' => 'mapsemble-' . $map->getMapId() ,
        'style' => 'overscroll-behaviour-y:auto;width:100%;height:' . $element['#height'] . ';',
      ],
    ];

    $element['script'] = [
      '#type' => 'html_tag',
      '#tag' => 'script',
      '#value' => 'window.addEventListener(\'DOMContentLoaded\', function () {

        window.Mapsemble.init({
          container: "#mapsemble-' . $map->getMapId() . '",
          mapId: "' . $map->getMapId() . '",
          embedHost: "' . Settings::get('mapsemble_app_base') . '",
          language: "' . \Drupal::languageManager()->getCurrentLanguage()->getId() . '",
          preventScale: true,
        });
      });',
    ];
    return $element;
  }

}
