<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\mapsemble\MapsembleMapInterface;

/**
 * Defines the Mapsemble map configuration entity class.
 *
 * @ConfigEntityType(
 *   id = "mapsemble_map",
 *   label = @Translation("Mapsemble map"),
 *   config_prefix = "map",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label"
 *   },
 *   handlers = {
 *     "list_builder" = "Drupal\mapsemble\MapsembleMapListBuilder",
 *     "form" = {
 *       "add" = "Drupal\mapsemble\Form\MapsembleMapForm",
 *       "edit" = "Drupal\mapsemble\Form\MapsembleMapForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm"
 *     }
 *   },
 *   links = {
 *     "collection" = "/admin/structure/mapsemble-map",
 *     "add-form" = "/admin/structure/mapsemble-map/add",
 *     "edit-form" = "/admin/structure/mapsemble-map/{mapsemble_map}",
 *     "configure-form" = "/admin/structure/mapsemble-map/{mapsemble_map}/configure",
 *     "delete-form" = "/admin/structure/mapsemble-map/{mapsemble_map}/delete"
 *   },
 *   admin_permission = "administer mapsemble",
 *   constraints = {
 *     "ImmutableProperties" = {"id"}
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "settings"
 *   }
 * )
 */
final class MapsembleMap extends ConfigEntityBase implements MapsembleMapInterface {

  /**
   * The map ID.
   *
   * @var string
   */
  protected string $id = '';

  /**
   * The map label.
   *
   * @var string
   */
  protected string $label = '';

  /**
   * The map configuration.
   *
   * @var array
   */
  protected array $settings = [];

  /**
   * Get the configuration for the map.
   *
   * @return array
   *   The map configuration.
   */
  public function getSettings(): array {
    return $this->settings;
  }

  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->label;
  }

  /**
   * Retrieves the ID of the layer.
   *
   * @return string
   *   The ID of the layer.
   */
  public function getLayerId(): string {
    return $this->settings['mapsemble']['layerId'] ?? '';
  }

  /**
   * Retrieves the ID of the map.
   *
   * @return string
   *   The ID of the map.
   */
  public function getMapId(): string {
    return $this->settings['mapsemble']['mapId'] ?? '';
  }

  /**
   * Retrieves the remoteSlug.
   */
  public function getRemoteSlug(): string {
    return $this->settings['mapsemble']['remoteSlug'] ?? '';
  }

}
