<?php

declare(strict_types=1);

namespace Drupal\mapsemble;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Renderer;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\Security\DoTrustedCallbackTrait;
use Drupal\Core\Site\Settings;
use Drupal\Core\State\State;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\geofield\GeoPHP\GeoPHPWrapper;
use Drupal\mapsemble\Entity\MapsembleMap;
use Drupal\user\Entity\User;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;

/**
 * Provides an API service for interacting with Mapsemble.
 */
final class MapsembleApi {

  use StringTranslationTrait;

  use DoTrustedCallbackTrait;

  /**
   * The map storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $mapStorage;

  /**
   * The base URL for the Mapsemble app.
   */
  const MAPSEMBLE_APP_BASE = 'https://app.mapsemble.com';

  const MAPSEMBLE_BASE = 'https://mapsemble.com';

  const MAPS_ENDPOINT = '/api/v1/maps';

  const DATA_ENDPOINT = '/api/v1/layer/%s/features';

  /**
   * Constructs a new MapsembleApi object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Render\Renderer $renderer
   *   The renderer service.
   * @param \Drupal\geofield\GeoPHP\GeoPHPWrapper $geoPHP
   *   The GeoPHP wrapper service.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $urlGenerator
   *   The URL generator service.
   * @param \Drupal\Core\State\State $state
   *   The state service used to store token information.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service for user notifications.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly Renderer $renderer,
    private readonly GeoPHPWrapper $geoPHP,
    private readonly UrlGeneratorInterface $urlGenerator,
    private readonly State $state,
    private readonly MessengerInterface $messenger,
  ) {
    $this->mapStorage = $this->entityTypeManager->getStorage('mapsemble_map');
  }

  /**
   * Get the token for making requests.
   */
  public function getToken() {
    $token = $this->state->get('mapsemble_token');
    // Add a 60-second margin to token expiration to ensure we don't use a token that's about to expire.
    $margin = 60;
    if ($token && $token['expires_at'] > (time() + $margin)) {
      return $token['access_token'];
    }
    else {
      $client = $this->getClient(FALSE);
      $client_id = $this->configFactory->get('mapsemble.settings')->get(
        'client_id'
      );
      $client_secret = $this->configFactory->get('mapsemble.settings')->get(
        'client_secret'
      );
      try {
        $request = $client->request('POST', '/token', [
          'form_params' => [
            'grant_type' => 'client_credentials_with_user',
            'client_id' => $client_id,
            'client_secret' => $client_secret,
          ],
        ]);
      }
      catch (GuzzleException $e) {
        $this->messenger->addError(
          'Could not get token from Mapsemble. Please check your API key and try again. Error: ' . $e->getMessage(
          )
        );
      }
      $content = json_decode($request->getBody()->getContents(), TRUE);
      $expires_in = $content['expires_in'];
      $expires_at = time() + $expires_in;
      $this->state->set('mapsemble_token', [
        'access_token' => $content['access_token'],
        'expires_at' => $expires_at,
      ]);

      return $content['access_token'];
    }
  }

  /**
   * Retrieves remote maps from Mapsemble.
   *
   * @return array
   *   An array of remote maps.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function getRemoteMaps(): array {
    $client = $this->getClient();
    $request = $client->request('GET', self::MAPS_ENDPOINT);
    $content = json_decode($request->getBody()->getContents(), TRUE);
    $maps = [];
    foreach ($content as $map) {
      foreach ($map['layers'] as $layer) {
        $maps[$map['id']] = $map;
        break;
      }
    }

    return $maps;
  }

  public function createDrupalMap(
    ?string $label = NULL,
    string $starterKit = 'drupal.storeLocator',
  ) {
    $client = $this->getClient();

    $payload = [
      'starterKit' => $starterKit,
    ];
    if (!empty($label)) {
      // Mapsemble API expects the map name under 'label'.
      $payload['label'] = $label;
    }

    $payload['baseUrl'] = \Drupal::request()->getSchemeAndHttpHost();

    $response = $client->post('/api/v1/maps', [
      RequestOptions::JSON => $payload,
    ]);

    return json_decode($response->getBody()->getContents(), TRUE);
  }

  /**
   * Retrieves local maps from Drupal.
   *
   * @return array
   *   An array of local maps.
   */
  public function getLocalMaps(): array {
    return $this->entityTypeManager->getStorage('mapsemble_map')->loadMultiple(
    );
  }

  /**
   * Retrieves a local map entity by its remote map ID.
   *
   * @param string $id
   *   The remote map ID.
   *
   * @return \Drupal\mapsemble\Entity\MapsembleMap
   *   The local map entity.
   */
  protected function getLocalMapEntityByRemoteMapId(string $id): MapsembleMap {
    $maps = $this->mapStorage->loadByProperties(['mapId' => $id]);
    $map = reset($maps);
    assert($map instanceof MapsembleMap);

    return $map;
  }

  /**
   * Retrieves data entities from a map.
   *
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map entity.
   * @param array $filters
   *   Optional array of filters to apply (from remote requests).
   *
   * @return array
   *   An array of data entities.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getDataEntitiesFromMap(
    MapsembleMap $map,
    array $filters = [],
  ): array {
    $entityTypeId = $map->get('settings')['entity_type_id'] ?? '';
    $bundle = $map->get('settings')['bundle'] ?? '';
    if (!$bundle || !$entityTypeId) {
      return [];
    }

    $entityStorage = $this->entityTypeManager->getStorage($entityTypeId);
    $definition = $this->entityTypeManager->getDefinition($entityTypeId);

    // Build base properties
    $properties = [];
    if ($definition->hasKey('bundle')) {
      $properties[$definition->getKey('bundle')] = $bundle;
    }

    // Apply remote filter mappings if filters are provided
    $filterMappings = $map->get('settings')['filter_mappings'] ?? [];
    if (!empty($filters) && !empty($filterMappings)) {
      foreach ($filters as $filterId => $filterValue) {
        // Check if this filter is mapped to a Drupal field
        if (isset($filterMappings[$filterId]) && !empty($filterMappings[$filterId])) {
          $drupalField = $filterMappings[$filterId];
          // Add the filter value to the properties for entity query
          $properties[$drupalField] = $filterValue;
        }
      }
    }

    return $entityStorage->loadByProperties($properties);
  }

  /**
   * Cache of existing features per map ID.
   *
   * @var array
   */
  protected array $existingContent = [];

  /**
   * Retrieves existing features from Mapsemble for a given map.
   *
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map entity whose features should be retrieved.
   *
   * @return array
   *   An array of features from the remote layer.
   */
  public function getExistingFeatures(MapsembleMap $map) {
    if (empty($this->existingContent[$map->id()])) {
      $client = $this->getClient();
      $response = $client->request(
        'GET',
        sprintf(self::DATA_ENDPOINT, $map->getLayerId())
      );
      $existingContent = json_decode($response->getBody()->getContents(), TRUE);
      $this->existingContent[$map->id(
      )] = $existingContent['featureCollection']['features'] ?? [];
    }

    return $this->existingContent[$map->id()];
  }

  /**
   * Callback function for batch processing completion.
   *
   * @param bool $success
   *   Indicates if the batch was successful.
   * @param array $results
   *   The results of the batch operations.
   * @param array $operations
   *   The operations that were processed.
   */
  public static function batchFinishedCallback(
    bool $success,
    array $results,
    array $operations,
  ): void {
    if ($success) {
      \Drupal::messenger()->addMessage(
        t(
          'Successfully syncronised content to mapsembls. Processed @count entities.',
          [
            '@count' => $results['processed'] ?? 0,
          ]
        )
      );
    }
    else {
      \Drupal::messenger()->addMessage(
        t('Batch did not complete. Some operations were not finished.')
      );
    }
  }

  /**
   * Synchronizes all entities for a specific map.
   *
   * @param \Drupal\mapsemble\Entity\MapsembleMap $mapsemble_map
   *   The map entity to synchronize.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response to the batch processing page.
   */
  public function buildBatchForSyncing(
    MapsembleMap $mapsemble_map,
  ): BatchBuilder {
    $batch_builder = new BatchBuilder();
    $batch_builder->setTitle(
      $this->t(
        'Synchronizing entities for map @label',
        ['@label' => $mapsemble_map->label()]
      )
    )
      ->setInitMessage($this->t('Starting synchronization...'))
      ->setProgressMessage($this->t('Synchronizing with Mapsemble...'))
      ->setErrorMessage($this->t('An error occurred during synchronization.'))
      ->setFinishCallback(
        [static::class, 'batchFinishedCallback'],
      );

    // Add operations to the batch.
    $existingFeatures = $this->getExistingFeatures($mapsemble_map);
    $entities = $this->getDataEntitiesFromMap($mapsemble_map);
    foreach ($entities as $entity) {
      $batch_builder->addOperation(
        [self::class, 'batchSyncEntity'],
        [$entity->id(), $entity->getEntityTypeId(), $mapsemble_map->id()]
      );
      foreach ($existingFeatures as $delta => $feature) {
        if (($feature['properties']['remote-id'] ?? NULL) == $entity->id()) {
          unset($existingFeatures[$delta]);
        }
      }
    }

    foreach ($existingFeatures as $feature) {
      $batch_builder->addOperation(
        [self::class, 'batchDeleteFeature'],
        [
          $feature['id'],
          $mapsemble_map->id(),
        ]
      );
    }

    return $batch_builder;
  }

  /**
   * Batch operation to synchronize a single entity.
   *
   * @param int $entity_id
   *   The ID of the entity to synchronize.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $map_id
   *   The ID of the map.
   * @param array $context
   *   The batch context.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public static function batchSyncEntity(
    int $entity_id,
    string $entity_type_id,
    string $map_id,
    array &$context,
  ): void {
    $entityTypeManager = \Drupal::entityTypeManager();
    $entity = $entityTypeManager->getStorage($entity_type_id)->load($entity_id);
    $map = $entityTypeManager->getStorage('mapsemble_map')->load($map_id);

    assert($map instanceof MapsembleMap);
    if ($entity && $map) {
      /** @var \Drupal\mapsemble\MapsembleApi $mapsemble_api */
      $mapsemble_api = \Drupal::service('mapsemble.api');

      // Determine if this operation will result in a deletion on Mapsemble.
      $anonymousUser = User::load(0);
      $delete = FALSE;
      if (!$entity->access('view', $anonymousUser)) {
        $delete = TRUE;
      }
      elseif ($entity instanceof EntityPublishedInterface && !$entity->isPublished(
        )) {
        $delete = TRUE;
      }

      $context['results']['processed'] = ($context['results']['processed'] ?? 0) + 1;

      // Report the appropriate batch message based on the action taken.
      if ($delete) {
        $mapsemble_api->deleteEntity($entity, $map);
        $context['message'] = t(
          'Deleting feature for entity ID @id from map ID @map_id',
          [
            '@id' => $entity_id,
            '@map_id' => $map_id,
          ]
        );
      }
      else {
        $mapsemble_api->syncEntity($entity, $map);
        $context['message'] = t(
          'Synchronizing entity ID @id for map ID @map_id',
          [
            '@id' => $entity_id,
            '@map_id' => $map_id,
          ]
        );
      }
    }
    else {
      // Fallback messages if required data is missing.
      if (!$map) {
        $context['message'] = t('No map found for ID @map_id', [
          '@map_id' => $map_id,
        ]);
      }
      elseif (!$entity) {
        $context['message'] = t('Entity ID @id not found; skipping', [
          '@id' => $entity_id,
        ]);
      }
    }
  }

  /**
   * Batch operation to delete a feature from Mapsemble.
   *
   * @param string $feature_id
   *   The ID of the feature to delete.
   * @param string $map_id
   *   The map configuration entity ID.
   * @param array $context
   *   The batch context.
   */
  public static function batchDeleteFeature(
    string $feature_id,
    string $map_id,
    array &$context,
  ): void {
    $entityTypeManager = \Drupal::entityTypeManager();
    $map = $entityTypeManager->getStorage('mapsemble_map')->load($map_id);
    assert($map instanceof MapsembleMap);
    if ($map) {
      /** @var \Drupal\mapsemble\MapsembleApi $mapsemble_api */
      $mapsemble_api = \Drupal::service('mapsemble.api');
      $mapsemble_api->deleteFeature($feature_id, $map);
      $context['results']['processed'] = ($context['results']['processed'] ?? 0) + 1;
      $context['message'] = t('Deleting feature @id', [
        '@id' => $feature_id,
      ]);
    }
    else {
      $context['message'] = t('No map found');
    }
  }

  /**
   * Checks whether an entity already exists in the remote content list.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check for.
   * @param array $existingContent
   *   The list of existing features from Mapsemble.
   *
   * @return bool
   *   TRUE if the entity exists remotely; otherwise FALSE.
   */
  private function inExistingContent($entity, $existingContent) {
    foreach ($existingContent as $existingFeature) {
      if ($existingFeature['properties']['remote-id'] === $entity->id()) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Gets a configured HTTP client.
   *
   * @return \GuzzleHttp\Client
   *   The HTTP client.
   */
  protected function getClient($use_token = TRUE): Client {
    $token = NULL;
    if ($use_token) {
      $token = $this->getToken();
    }

    return new Client([
      'verify' => Settings::get(
        'mapsemble_app_base'
      ) ? FALSE : TRUE,

      'base_uri' => Settings::get(
          'mapsemble_app_base'
      ) ?? self::MAPSEMBLE_APP_BASE,
      'headers' => $use_token ? [
        'Authorization' => 'Bearer ' . $token,
        'Accept' => 'application/json',
      ] : [],
    ]);
  }

  /**
   * Deletes an entity from Mapsemble.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to delete.
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map entity.
   */
  public function deleteEntity(
    EntityInterface $entity,
    MapsembleMap $map,
  ): void {
    try {
      $existingContent = $this->getExistingFeatures($map);
      foreach ($existingContent as $existingFeature) {
        if ($existingFeature['properties']['remote-id'] === $entity->id()) {
          $this->deleteFeature($existingFeature['id'], $map);
        }
      }
    }
    catch (\Exception) {
      // Do nothing.
    }
  }

  /**
   * Deletes a feature from Mapsemble.
   *
   * @param string $id
   *   The feature ID to delete.
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map configuration entity.
   */
  public function deleteFeature(
    string $id,
    MapsembleMap $map,
  ): void {
    $client = $this->getClient();
    try {
      $client->request(
        'DELETE',
        sprintf(self::DATA_ENDPOINT, $map->getLayerId()),
        [
          'json' => [
            'type' => 'FeatureCollection',
            'features' => [
              [
                'type' => 'Feature',
                'id' => $id,
              ],
            ],
          ],
        ]
      );
    }
    catch (\Exception) {
      // Do nothing.
    }
  }

  /**
   * Gets the API key.
   *
   * @return string
   *   The API key.
   */
  protected function getApiKey(): string {
    return $this->configFactory->get('mapsemble.settings')->get('api_key');
  }

  /**
   * Gets the Mapsemble content URL.
   *
   * @param string $routeName
   *   The route name.
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map entity.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string
   *   The content URL.
   */
  protected function getMapsembleContentUrl(
    string $routeName,
    MapsembleMap $map,
    EntityInterface $entity,
  ): string {
    return $this->urlGenerator->generate(
      $routeName,
      [
        'mapsemble_map' => $map->id(),
        'entity_type' => $entity->getEntityTypeId(),
        'entity' => $entity->id(),
      ],
      UrlGeneratorInterface::ABSOLUTE_URL
    );
  }

  /**
   * Gets the remote feature ID for an entity, if it exists.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The local entity.
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map configuration entity.
   *
   * @return string|null
   *   The feature ID if found; otherwise NULL.
   */
  public function getFeatureId(EntityInterface $entity, MapsembleMap $map) {
    $content = $this->getExistingFeatures($map);
    foreach ($content as $existingFeature) {
      if (($existingFeature['properties']['remote-id'] ?? NULL) === $entity->id(
        )) {
        return $existingFeature['id'];
      }
    }

    return NULL;
  }

  /**
   * Syncs a single entity to Mapsemble.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to sync.
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The map entity.
   */
  public function syncEntity(
    EntityInterface $entity,
    MapsembleMap $map,
  ): void {
    if (!$map->getMapId()) {
      return;
    }
    if (!$map->getLayerId()) {
      return;
    }
    assert($entity instanceof ContentEntityInterface);
    $client = $this->getClient();
    $location_field = $map->get('settings')['location_field'] ?? '';

    if (!$location_field) {
      return;
    }
    if (!$entity->hasField($location_field)) {
      return;
    }

    if (!$entity->get($location_field)->value) {
      return;
    }

    $geometry = $this->geoPHP->load($entity->get($location_field)->value, 'wkt')
      ->out(
        'json'
      );
    if (!$geometry) {
      return;
    }

    $feature = [
      'type' => 'Feature',
      'id' => $this->getFeatureId($entity, $map),
      'geometry' => json_decode(
        $geometry,
        TRUE
      ),
      'properties' => [
        'label' => $entity->label(),
        'remote-id' => $entity->id(),
      ],
    ];

    // Add mapped fields to properties if configured.
    $settings = $map->get('settings') ?? [];
    $field_mappings = $settings['field_mappings'] ?? [];
    $field_formatters = $settings['field_formatters'] ?? [];

    if (!empty($field_mappings) && is_array($field_mappings)) {
      foreach ($field_mappings as $target_slug => $source_field_name) {
        if (empty($source_field_name)) {
          continue;
        }
        if (!$entity->hasField($source_field_name)) {
          continue;
        }
        $items = $entity->get($source_field_name);
        if ($items->isEmpty()) {
          continue;
        }

        $value = NULL;

        // If a formatter is configured for this target slug, try to render with it.
        $formatter_conf = $field_formatters[$target_slug] ?? [];
        $plugin_id = $formatter_conf['plugin'] ?? '';
        $plugin_settings = $formatter_conf['settings'] ?? [];
        if (!empty($plugin_id)) {
          try {
            /** @var \Drupal\Core\Field\FormatterPluginManager $formatter_manager */
            $formatter_manager = \Drupal::service('plugin.manager.field.formatter');
            $formatter = $formatter_manager->createInstance($plugin_id, [
              'field_definition' => $items->getFieldDefinition(),
              'settings' => $plugin_settings,
              'view_mode' => 'full',
              'third_party_settings' => [],
              'label' => 'hidden',
            ]);
            // Ensure prepareView is executed so formatters (e.g., entity reference label)
            // can pre-load required data before rendering.
            $formatter->prepareView([$entity->id() => $items]);
            $elements = $formatter->viewElements($items, $entity->language()->getId());
            // Render to plain text to store in JSON properties.
            $rendered = $this->renderer->renderPlain($elements);
            $value = trim(strip_tags((string) $rendered));
          }
          catch (\Throwable) {
            // If formatter fails, fall back to extracting raw values.
            $value = NULL;
          }
        }

        if ($value === NULL || $value === '') {
          // Fallback to raw extraction: join multiple items as comma-separated string.
          $parts = [];
          foreach ($items as $delta => $item) {
            // Prefer getString if available.
            if (method_exists($item, 'getString')) {
              $str = $item->getString();
              if ($str !== '') {
                $parts[] = $str;
                continue;
              }
            }
            // Try common properties.
            if (isset($item->value)) {
              $parts[] = (string) $item->value;
            }
            elseif (isset($item->target_id)) {
              // Entity reference: use label if possible.
              try {
                $target = $item->entity ?? NULL;
                if ($target && method_exists($target, 'label')) {
                  $parts[] = (string) $target->label();
                }
                else {
                  $parts[] = (string) $item->target_id;
                }
              }
              catch (\Throwable) {
                $parts[] = (string) $item->target_id;
              }
            }
          }
          $value = trim(implode(', ', array_filter($parts, static function ($v) {
            return $v !== '' && $v !== NULL;
          })));
        }

        if ($value !== '') {
          $feature['properties'][$target_slug] = $value;
        }
      }
    }

    try {
      $response = $client->request(
        'PUT',
        sprintf(self::DATA_ENDPOINT, $map->getLayerId()),
        [
          'json' => [
            'type' => 'FeatureCollection',
            'features' => [$feature],
          ],
        ]
      );
      $content = json_decode($response->getBody()->getContents(), TRUE);
      if (!($content['success'] ?? NULL)) {
        \Drupal::logger('mapsemble')->error(
          'Error syncing entity: ' . $content['failed'][0]['message']
        );
      }
    }
    catch (\Exception) {
      $foo = 'bar';
    }
  }

}
