<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Plugin\Field\FieldWidget;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Widget that allows selecting a Mapsemble map entity.
 */
#[FieldWidget(
  id: 'mapsemble_map_select',
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup('Mapsemble map select'),
  field_types: ['mapsemble_map']
)]
final class MapsembleMapSelectWidget extends WidgetBase {

  public function __construct(
    string $plugin_id,
    mixed $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(\Drupal\Core\Field\FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $selected = $items[$delta]->get('map_id')->getString() ?? '';

    $options = ['' => '- Select -'];
    $maps = $this->entityTypeManager->getStorage('mapsemble_map')->loadMultiple();

    // Sort by label.
    uasort($maps, static fn($a, $b) => strnatcasecmp($a->label(), $b->label()));

    foreach ($maps as $map) {
      $options[$map->id()] = $map->label();
    }

    $element['map_id'] = $element + [
      '#type' => 'select',
      '#title' => $this->t('Map'),
      '#options' => $options,
      '#default_value' => $selected,
      '#required' => $this->fieldDefinition->isRequired(),
    ];

    return $element;
  }

}
