<?php

namespace Drupal\mapsemble\Plugin\Filter;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mapsemble\Attribute\FilterPlugin;
use Drupal\mapsemble\Plugin\FilterPluginBase;

/**
 * Filter plugin for taxonomy reference fields.
 */
#[FilterPlugin(
  id: "taxonomy_field_filter",
  label: new TranslatableMarkup("Taxonomy Field Filter"),
  field_types: [
    "entity_reference",
  ],
)]
class TaxonomyFieldFilter extends FilterPluginBase {

  /**
   * {@inheritdoc}
   */
  public function filter(string $entityTypeId, string $bundle, string $fieldName, FieldDefinitionInterface $fieldDefinition, string|array $value = ''): array {
    // Check if this is a taxonomy reference field.
    $settings = $fieldDefinition->getSettings();
    if (!isset($settings['target_type']) || $settings['target_type'] !== 'taxonomy_term') {
      // Not a taxonomy field, return empty array.
      return [];
    }

    $entityStorage = $this->entityTypeManager->getStorage($entityTypeId);
    $definition = $this->entityTypeManager->getDefinition($entityTypeId);

    // Build base query.
    $query = $entityStorage->getQuery()
      ->condition('status', 1)
      ->accessCheck(FALSE);

    if ($definition->hasKey('bundle')) {
      $query->condition($definition->getKey('bundle'), $bundle);
    }

    // Only get entities that have a value for this field.
    $query->exists($fieldName);

    // Filter by value if provided.
    if (!empty($value)) {
      $vocabularies = $settings['handler_settings']['target_bundles'] ?? [];

      if (!empty($vocabularies)) {
        $termStorage = $this->entityTypeManager->getStorage('taxonomy_term');
        $matchingTermIds = [];

        foreach ($vocabularies as $vocabulary) {
          // Search for terms by ID.
          $termQuery = $termStorage->getQuery()
            ->condition('vid', $vocabulary)
            ->condition('tid', $value)
            ->accessCheck(FALSE);
          $termIds = $termQuery->execute();
          $matchingTermIds = array_merge($matchingTermIds, $termIds);
        }

        if (!empty($matchingTermIds)) {
          $query->condition($fieldName, $matchingTermIds, 'IN');
        }
        else {
          // No matching terms, return empty array.
          return [];
        }
      }
    }

    $entityIds = $query->execute();

    // Return entity IDs as array values (not associative).
    return array_values($entityIds);
  }

}
