<?php

declare(strict_types=1);

namespace Drupal\mapsemble_store_locator_demo\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\mapsemble_store_locator_demo\StoreInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the store entity class.
 *
 * @ContentEntityType(
 *   id = "mapsemble_store",
 *   label = @Translation("Mapsemble Store"),
 *   label_collection = @Translation("Mapsemble Stores"),
 *   label_singular = @Translation("Mapsemble store"),
 *   label_plural = @Translation("Mapsemble stores"),
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid"
 *   },
 *   handlers = {
 *     "list_builder" = "Drupal\mapsemble_store_locator_demo\StoreListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "access" = "Drupal\mapsemble_store_locator_demo\StoreAccessControlHandler",
 *     "form" = {
 *       "add" = "Drupal\mapsemble_store_locator_demo\Form\StoreForm",
 *       "edit" = "Drupal\mapsemble_store_locator_demo\Form\StoreForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm"
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\DefaultHtmlRouteProvider"
 *     }
 *   },
 *   links = {
 *     "collection" = "/admin/content/mapsemble-store",
 *     "add-form" = "/admin/content/mapsemble/stores/add",
 *     "canonical" = "/admin/content/mapsemble/stores/{mapsemble_store}",
 *     "edit-form" = "/admin/content/mapsemble/stores/{mapsemble_store}/edit",
 *     "delete-form" = "/admin/content/mapsemble/stores/{mapsemble_store}/delete",
 *     "delete-multiple-form" = "/admin/content/mapsemble-store/delete-multiple"
 *   },
 *   admin_permission = "administer mapsemble_store",
 *   base_table = "mapsemble_store",
 *   label_count = {
 *     "singular" = "@count store",
 *     "plural" = "@count stores"
 *   },
 *   field_ui_base_route = "entity.mapsemble_store.settings"
 * )
 */
final class Store extends ContentEntityBase implements StoreInterface {
  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the store was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the store was last edited.'));

    return $fields;
  }

}
