<?php

declare(strict_types=1);

namespace Drupal\mapsemble_store_locator_demo\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\mapsemble\MapsembleApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to manually create the Mapsemble map and synchronize content.
 */
final class MapsembleStoreLocatorSetupForm extends FormBase {

  public function __construct(
    private readonly MapsembleApi $mapsembleApi,
  ) {}

  public static function create(ContainerInterface $container): static {
    $instance = new static(
      $container->get('mapsemble.api'),
    );
    // Use MessengerTrait's setter to avoid property redeclaration conflicts.
    if (method_exists($instance, 'setMessenger')) {
      $instance->setMessenger($container->get('messenger'));
    }
    return $instance;
  }

  public function getFormId(): string {
    return 'mapsemble_store_locator_setup_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state): array {
    $account = $this->currentUser();

    if (!$account->hasPermission('administer mapsemble')) {
      // No access to the button, but still return an empty form structure.
      $form['message'] = [
        '#markup' => $this->t('You do not have permission to administer Mapsemble.'),
      ];
      return $form;
    }

    // Show info about current configuration status.
    $config = $this->config('mapsemble.settings');
    $client_id = (string) $config->get('client_id');
    $client_secret = (string) $config->get('client_secret');

    if (empty($client_id) || empty($client_secret)) {
      $settings_url = Url::fromRoute('mapsemble.settings');
      $settings_link = Link::fromTextAndUrl($this->t('Mapsemble settings page'), $settings_url)->toString();
      $form['keys_warning'] = [
        '#type' => 'container',
        'text' => [
          '#markup' => '<div class="messages messages--warning">' . $this->t('Mapsemble API credentials are missing. Please configure the Client ID and Client Secret on the @settings_link before proceeding. Alternatively, add them to your settings.local.php (preferred for local/dev) as follows:<br/><br/><code>$config["mapsemble.settings"]["client_id"] = "your-client-id";<br/>$config["mapsemble.settings"]["client_secret"] = "your-client-secret";</code>', [
            '@settings_link' => $settings_link,
          ]) . '</div>',
        ],
      ];
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['create_sync'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create Map in Mapsemble and Synchronize Content'),
      '#button_type' => 'primary',
      '#disabled' => (empty($client_id) || empty($client_secret)),
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Validate token before attempting any actions.
    try {
      $token = $this->mapsembleApi->getToken();
      if (empty($token)) {
        $this->messenger->addError($this->t('Could not obtain a token from Mapsemble. Please verify your API credentials.'));
        return;
      }
    }
    catch (\Throwable $e) {
      $this->messenger->addError($this->t('Could not obtain a token from Mapsemble. Please verify your API credentials. Error: @msg', ['@msg' => $e->getMessage()]));
      return;
    }

    // Create the remote map and update local config via existing installer logic.
    try {
      /** @var \Drupal\mapsemble_store_locator_demo\MapsembleStoreLocatorInstaller $installer */
      $installer = \Drupal::service('mapsemble_store_locator_demo.installer');
      $installer->createMapsembleMapOnInstall();
    }
    catch (\Throwable $e) {
      $this->messenger->addError($this->t('Failed to create the Mapsemble map: @msg', ['@msg' => $e->getMessage()]));
      return;
    }

    // Trigger batch synchronization for the demo map entity (store_locator).
    try {
      /** @var \Drupal\mapsemble\Entity\MapsembleMap $map */
      $map = \Drupal::entityTypeManager()->getStorage('mapsemble_map')->load('store_locator');
      if ($map) {
        $batch_builder = $this->mapsembleApi->buildBatchForSyncing($map);
        batch_set($batch_builder->toArray());
        $this->messenger->addStatus($this->t('The Mapsemble map was created. Synchronization has started.'));
      }
      else {
        $this->messenger->addWarning($this->t('The local map entity "store_locator" could not be found.'));
      }
    }
    catch (\Throwable $e) {
      $this->messenger->addError($this->t('Failed to start synchronization: @msg', ['@msg' => $e->getMessage()]));
    }
  }
}
