<?php

namespace Drupal\mapsemble\Event;

use Drupal\mapsemble\Entity\MapsembleMap;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Event that is fired when a filter is applied to a Mapsemble map.
 */
class MapsembleFilterEvent extends Event {

  /**
   * The event name.
   */
  const EVENT_NAME = 'mapsemble.filter';


  /**
   * The filtered entity IDs.
   *
   * @var array|null
   */
  protected $entityIds = NULL;

  /**
   * Constructs a MapsembleFilterEvent object.
   *
   * @param \Drupal\mapsemble\Entity\MapsembleMap $map
   *   The MapsembleMap entity.
   * @param string $filterName
   *   The filter name.
   * @param string $value
   *   The search value.
   */
  public function __construct(
    protected MapsembleMap $map,
    protected string $filterName,
    protected array|string $value = '',
  ) {
  }

  /**
   * Gets the MapsembleMap entity.
   *
   * @return \Drupal\mapsemble\Entity\MapsembleMap
   *   The MapsembleMap entity.
   */
  public function getMap(): MapsembleMap {
    return $this->map;
  }

  /**
   * Gets the filter name.
   *
   * @return string
   *   The filter name.
   */
  public function getFilterName(): string {
    return $this->filterName;
  }

  /**
   * Gets the search value.
   *
   * @return string
   *   The search value.
   */
  public function getValue(): array|string {
    return $this->value;
  }

  /**
   * Gets the filtered entity IDs.
   *
   * @return array|null
   *   The filtered entity IDs, or NULL if not set.
   */
  public function getEntityIds(): ?array {
    return $this->entityIds;
  }

  /**
   * Sets the filtered entity IDs.
   *
   * @param array $entityIds
   *   The filtered entity IDs.
   *
   * @return $this
   */
  public function setEntityIds(array $entityIds): self {
    $this->entityIds = $entityIds;
    return $this;
  }

  /**
   * Checks if filtering has been applied.
   *
   * @return bool
   *   TRUE if filtering has been applied, FALSE otherwise.
   */
  public function hasEntityIds(): bool {
    return $this->entityIds !== NULL;
  }

}
