<?php

namespace Drupal\mapsemble\EventSubscriber;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\mapsemble\Event\MapsembleFilterEvent;
use Drupal\mapsemble\Plugin\FilterPluginManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for filter events.
 */
class FilterEventSubscriber implements EventSubscriberInterface {

  /**
   * The filter plugin manager.
   *
   * @var \Drupal\mapsemble\Plugin\FilterPluginManager
   */
  protected $filterPluginManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs a new FilterEventSubscriber object.
   *
   * @param \Drupal\mapsemble\Plugin\FilterPluginManager $filter_plugin_manager
   *   The filter plugin manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   */
  public function __construct(FilterPluginManager $filter_plugin_manager, EntityFieldManagerInterface $entity_field_manager) {
    $this->filterPluginManager = $filter_plugin_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      MapsembleFilterEvent::EVENT_NAME => ['onFilter', 0],
    ];
  }

  /**
   * Reacts to the filter event.
   *
   * @param \Drupal\mapsemble\Event\MapsembleFilterEvent $event
   *   The filter event.
   */
  public function onFilter(MapsembleFilterEvent $event) {
    // If filtering has already been applied by another subscriber, skip.
    if ($event->hasEntityIds()) {
      return;
    }

    $map = $event->getMap();
    $filterName = $event->getFilterName();
    $value = $event->getValue();

    // Get the filter mappings from the map settings.
    $filterMappings = $map->get('settings')['filter_mappings'] ?? [];
    if (!isset($filterMappings[$filterName])) {
      // Filter not mapped, skip.
      return;
    }

    $fieldName = $filterMappings[$filterName];
    if (empty($fieldName)) {
      // Filter not mapped to a field, skip.
      return;
    }

    // Get entity type and bundle from map settings.
    $entityTypeId = $map->get('settings')['entity_type_id'] ?? '';
    $bundle = $map->get('settings')['bundle'] ?? '';
    if (!$entityTypeId || !$bundle) {
      // Map not properly configured, skip.
      return;
    }

    // Get the field definition.
    $bundleFields = $this->entityFieldManager->getFieldDefinitions($entityTypeId, $bundle);
    if (!isset($bundleFields[$fieldName])) {
      // Field not found, skip.
      return;
    }

    $fieldDefinition = $bundleFields[$fieldName];
    $fieldType = $fieldDefinition->getType();

    // Get the appropriate plugin for this field type.
    $plugin = $this->filterPluginManager->getPluginForFieldType($fieldType);
    if (!$plugin) {
      // No plugin found for this field type, skip.
      return;
    }

    // Execute the filtering.
    $entityIds = $plugin->filter($entityTypeId, $bundle, $fieldName, $fieldDefinition, $value);

    // Set the entity IDs on the event.
    $event->setEntityIds($entityIds);
  }

}
