<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Form;

use Drupal\mapsemble\Plugin\FieldPluginManager;
use Drupal\mapsemble\Plugin\FilterPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Field\FormatterPluginManager;
use Drupal\Core\Link;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\mapsemble\Entity\MapsembleMap;
use Drupal\mapsemble\MapsembleApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Map form.
 */
final class MapsembleMapForm extends EntityForm {

  /**
   * The remote maps.
   *
   * @var array
   */
  protected array $remoteMaps;

  /**
   * The selected remote map.
   *
   * @var array
   */
  protected array $selectedRemoteMap = [];

  /**
   * Creates an instance of the form.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container.
   *
   * @return static
   *   The form instance.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('mapsemble.api'),
      $container->get('plugin.manager.mapsemble_filter'),
      $container->get('plugin.manager.mapsemble_field'),
      $container->get('plugin.manager.field.formatter')
    );
  }

  /**
   * Constructs a new MapsembleMapForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   The entity display repository.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $bundleInfo
   *   The bundle info service.
   * @param \Drupal\mapsemble\MapsembleApi $mapsembleApi
   *   The Mapsemble API service.
   * @param \Drupal\mapsemble\Plugin\FilterPluginManager $filterPluginManager
   *   The filter plugin manager.
   * @param \Drupal\mapsemble\Plugin\FieldPluginManager $fieldPluginManager
   *   The field plugin manager.
   * @param \Drupal\Core\Field\FormatterPluginManager $formatterPluginManager
   *   The formatter plugin manager.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    EntityTypeManagerInterface $entityTypeManager,
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected EntityFieldManagerInterface $entityFieldManager,
    protected EntityTypeBundleInfoInterface $bundleInfo,
    protected MapsembleApi $mapsembleApi,
    protected FilterPluginManager $filterPluginManager,
    protected FieldPluginManager $fieldPluginManager,
    protected FormatterPluginManager $formatterPluginManager,
  ) {
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);

    // Toggle access based on the map selection value. When the special
    // '__create__' option is selected, hide submit/delete actions.
    $selected_map = $form_state->getValue(['settings', 'mapsemble', 'mapId']
    ) ?? $this->entity->getMapId();
    $show_actions = ($selected_map !== '__create__');

    if (isset($actions['submit'])) {
      $actions['submit']['#access'] = $show_actions;
    }
    if (isset($actions['delete'])) {
      $actions['delete']['#access'] = $show_actions;
    }

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    assert($this->entity instanceof MapsembleMap);
    $form = parent::form($form, $form_state);

    // Wrap the entire form so AJAX can refresh actions and settings together.
    $form['#prefix'] = '<div id="mapsemble-map-form-wrapper">';
    $form['#suffix'] = '</div>';

    $clientId = $this->configFactory->get('mapsemble.settings')->get(
      'client_id'
    );
    $clientSecret = $this->configFactory->get('mapsemble.settings')->get(
      'client_secret'
    );
    if (!$clientId || !$clientSecret) {
      $credentialsLink = Link::fromTextAndUrl(
        new TranslatableMarkup('here'),
        Url::fromRoute('mapsemble.settings')
      );
      $createlink = Link::fromTextAndUrl(
        new TranslatableMarkup('create a map'),
        Url::fromUri(
          (Settings::get(
            'mapsemble_app_base'
          ) ?? MapsembleApi::MAPSEMBLE_APP_BASE) . '/integrations/drupal',
          [
            'attributes' => [
              'target' => '_blank',
            ],
          ]
        )
      );
      $this->messenger()->addWarning(
        $this->t(
          'Please @create_link layer in Mapsemble first. Also configure your API credentials @credentials_link.',
          [
            '@create_link' => $createlink->toString(),
            '@credentials_link' => $credentialsLink->toString(),
          ]
        )
      );

      return $form;
    }
    else {
      try {
        $this->remoteMaps = $this->mapsembleApi->getRemoteMaps();
      }
      catch (\Exception $e) {
        $this->messenger()->addError(
          $this->t(
            'Error fetching remoteMaps from Mapsemble. Error: @error',
            ['@error' => $e->getMessage()]
          )
        );

        return $form;
      }

      if (!$this->remoteMaps) {
        $link = Link::fromTextAndUrl(
          new TranslatableMarkup('create a map'),
          Url::fromUri(
            (Settings::get(
              'mapsemble_app_base'
            ) ?? MapsembleApi::MAPSEMBLE_APP_BASE) . '/integrations/drupal',
            [
              'attributes' => [
                'target' => '_blank',
              ],
            ]
          )
        );
        $this->messenger()->addWarning(
          $this->t(
            'No remoteMaps found with a Drupal layer. First @link in Mapsemble with a Drupal layer. If necessary, create an account first.',
            [
              '@link' => $link->toString(),
            ]
          )
        );

        return $form;
      }
    }
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#machine_name' => [
        'exists' => [MapsembleMap::class, 'load'],
      ],
      '#disabled' => !$this->entity->isNew(),
      '#default_value' => $this->entity->id(),
    ];

    $form['messages'] = [
      '#theme' => 'status_messages',
      '#message_list' => [],
    ];

    // Filter mapping section
    $mapId = $form_state->getValue(['settings', 'mapsemble', 'mapId']
    ) ?? $this->entity->getMapId();
    $remoteFilters = [];
    if ($mapId && isset($this->remoteMaps[$mapId])) {
      $this->selectedRemoteMap = $this->remoteMaps[$mapId];
      // Look for filters in the map's layers with dataSource
      foreach ($this->selectedRemoteMap['layers'] ?? [] as $layer) {
        if (isset($layer['config']['filters'])) {
          foreach ($layer['config']['filters'] as $filter) {
            // Only include filters with handler 'remote'
            if (isset($filter['handler']) && $filter['handler'] === 'remote') {
              $remoteFilters[$filter['slug']] = $filter;
            }
          }
        }
      }
    }

    if ($this->selectedRemoteMap) {
      $remote_field = $this->selectedRemoteMap['layers'][1]['dataSource']['config']['remoteField'] ?? NULL;
      $form['mapsemble']['remoteIdSlug'] = [
        '#type' => 'hidden',
        '#value' => $remote_field,
      ];
      if (empty($remote_field)) {
        // Warn the user that the remote field is not configured on the Mapsemble map.
        $this->messenger()->addWarning(
          $this->t(
            'The selected Mapsemble map does not have its remote field configured. Please configure the remote field on the map in Mapsemble. You will not be able to save this form until it is set.'
          )
        );
      }
    }

    $form['mapsemble']['#prefix'] = '<div id="layer-wrapper">';
    $form['mapsemble']['#suffix'] = '</div>';

    // Mapsemble field group with Map ID selection.
    $form['mapsemble_group'] = [
      '#type' => 'details',
      '#title' => $this->t('Mapsemble'),
      '#description' => $this->t('Select the Mapsemble map you want to use.'),
      '#open' => TRUE,
    ];

    // Map select inside the Mapsemble group but keep data under settings[mapsemble][mapId].
    $form['mapsemble_group']['mapId'] = [
      '#type' => 'select',
      '#title' => $this->t('Map'),
      '#options' => (function () {
        $options = array_map(fn($map) => $map['label'], $this->remoteMaps);
        $options['__create__'] = $this->t('- Create new map -');

        return $options;
      })(),
      '#default_value' => $form_state->getValue(
          ['settings', 'mapsemble', 'mapId']
      ) ?? $this->entity->getMapId(),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::refreshEntityConfig',
        'wrapper' => 'mapsemble-map-form-wrapper',
      ],
      '#parents' => ['settings', 'mapsemble', 'mapId'],
    ];

    // Create map button, shown only when the create option is selected.
    $form['mapsemble_group']['create_map'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create map in mapsemble'),
      '#ajax' => [
        'callback' => '::refreshEntityConfig',
        'wrapper' => 'mapsemble-map-form-wrapper',
      ],
      '#submit' => ['::createNewMap'],
      '#limit_validation_errors' => [
        ['label'],
        ['settings', 'mapsemble', 'mapId'],
      ],
      '#access' => ($mapId === '__create__'),
    ];

    // Hidden values preserved under settings[mapsemble][...].
    $form['mapsemble_group']['layerId'] = [
      '#type' => 'hidden',
      '#value' => $this->selectedRemoteMap['layers'][1]['id'] ?? NULL,
      '#parents' => ['settings', 'mapsemble', 'layerId'],
    ];
    $form['mapsemble_group']['remoteSlug'] = [
      '#type' => 'hidden',
      '#value' => $this->selectedRemoteMap['layers'][1]['dataSource']['config']['remoteField'] ?? NULL,
      '#parents' => ['settings', 'mapsemble', 'remoteSlug'],
    ];

    // Drupal field group: keep all other fields here.
    $form['settings'] = [
      '#type' => 'details',
      '#prefix' => '<div id="mapsemble-map-configure-form">',
      '#suffix' => '</div>',
      '#title' => $this->t('Drupal'),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#access' => ($mapId !== '__create__'),
      'entity_type_id' => [
        '#type' => 'select',
        '#title' => $this->t('Entity type'),
        '#default_value' => $this->entity->get(
            'settings'
        )['entity_type_id'] ?? '',
        '#options' => $this->getEntityTypeOptions(),
        '#ajax' => [
          'callback' => '::refreshEntityConfig',
          'wrapper' => 'mapsemble-map-form-wrapper',
        ],
        '#empty_option' => $this->t('- Select -'),
        '#required' => TRUE,
        '#access' => ($mapId !== '__create__'),
      ],
    ];

    $entity_type_id = $form_state->getValue([
      'settings',
      'entity_type_id',
    ]) ?? $this->entity->get('settings')['entity_type_id'] ?? '';

    if ($entity_type_id) {
      $viewModes = $this->entityDisplayRepository->getViewModes(
        $entity_type_id

      );
      $form['settings']['bundle'] = [
        '#type' => 'select',
        '#title' => $this->t('Bundle'),
        '#default_value' => $this->entity->get('settings')['bundle'] ?? '',
        '#options' => array_map(function ($bundle) {
          return $bundle['label'];
        }, $this->bundleInfo->getBundleInfo($entity_type_id)),
        '#ajax' => [
          'callback' => '::refreshEntityConfig',
          'wrapper' => 'mapsemble-map-form-wrapper',
        ],
        '#empty_option' => $this->t('- Select -'),
        '#access' => ((bool) $entity_type_id) && ($mapId !== '__create__'),
        '#required' => TRUE,
      ];

      $bundle = $form_state->getValue([
        'settings',
        'bundle',
      ]) ?? $this->entity->get('settings')['bundle'] ?? '';

      $fields = [];
      if ($bundle) {
        $fields = $this->entityFieldManager->getFieldDefinitions(
          $entity_type_id,
          $bundle
        );
        foreach ($fields as $field) {
          if ($field->getType() !== 'geofield') {
            unset($fields[$field->getName()]);
          }
        }
      }

      if ($bundle && empty($fields)) {
        $this->messenger()->addWarning(
          $this->t(
            'No geofield found for the selected entity type and bundle. Please select a different entity.'
          )
        );
      }

      $form['settings']['location_field'] = [
        '#type' => 'select',
        '#title' => $this->t('Location field'),
        '#default_value' => $this->entity->get(
            'settings'
        )['location_field'] ?? '',
        '#ajax' => [
          'callback' => '::refreshEntityConfig',
          'wrapper' => 'mapsemble-map-form-wrapper',
        ],
        '#options' => array_map(
          fn($field) => $field->label(),
          $fields ?? []
        ),
        '#empty_option' => $this->t('- Select -'),
        '#required' => TRUE,
        '#access' => ((bool) $bundle) && ($mapId !== '__create__'),
      ];

      $location_field = $form_state->getValue([
        'settings',
        'location_field',
      ]) ?? $this->entity->get('settings')['location_field'] ?? '';

      $form['settings']['popup_view_mode'] = [
        '#type' => 'select',
        '#title' => $this->t('Popup view mode'),
        '#description' => $this->t(
          'The view mode to use for the popup in Mapsemble.'
        ),
        '#default_value' => $this->entity->get(
            'settings'
        )['popup_view_mode'] ?? '',
        '#options' => array_map(
          fn($view_mode) => $view_mode['label'],
          $viewModes
        ),
        '#empty_option' => $this->t('- Select -'),
        '#required' => TRUE,
        '#access' => ((bool) $location_field) && ($mapId !== '__create__'),
      ];
      $form['settings']['card_view_mode'] = [
        '#type' => 'select',
        '#title' => $this->t('Card view mode'),
        '#description' => $this->t(
          'The view mode to use for the card view in Mapsemble.'
        ),
        '#default_value' => $this->entity->get(
            'settings'
        )['card_view_mode'] ?? '',
        '#options' => array_map(
          fn($view_mode) => $view_mode['label'],
          $viewModes
        ),
        '#empty_option' => $this->t('- Select -'),
        '#required' => TRUE,
        '#access' => ((bool) $location_field) && ($mapId !== '__create__'),
      ];

      if ($bundle) {
        $form['settings']['filter_mappings'] = [
          '#type' => 'details',
          '#title' => $this->t('Filter by Drupal'),
          '#description' =>
          $this->t(
              'Let Drupal handle filters configured in Mapsemble.
            Select a field for each filter that can be used for filtering or
            provide custom filtering by creating an EventSubscriber for the
            MapsembleFilterEvent.'
          ),
          '#open' => TRUE,
          '#tree' => TRUE,
          '#access' => ($mapId !== '__create__'),
        ];

        // Get all fields for the selected bundle (not just geofields)
        $allFields = $this->entityFieldManager->getFieldDefinitions(
          $entity_type_id,
          $bundle
        );
        $fieldOptions = [];
        foreach ($allFields as $fieldName => $fieldDefinition) {
          // Skip base fields that aren't useful for filtering
          if (in_array(
            $fieldName,
            ['uuid', 'revision_log', 'revision_user', 'revision_created']
          )) {
            continue;
          }
          // Only include fields that have an applicable filter plugin.
          $fieldType = $fieldDefinition->getType();
          $plugin = $this->filterPluginManager->getPluginForFieldType(
            $fieldType
          );
          if ($plugin) {
            $fieldOptions[$fieldName] = $fieldDefinition->getLabel();
          }
        }

        if (empty($remoteFilters)) {
          // Show a warning inside the filter_mappings details when no remote filters are available.
          $form['settings']['filter_mappings']['_no_remote_filters'] = [
            '#type' => 'item',
            '#markup' => $this->t(
              "No filters with handler 'remote' were found in the selected Mapsemble map. Create filters in Mapsemble that use the 'remote' handler to map them to Drupal fields."
            ),
            '#prefix' => '<div class="messages messages--warning">',
            '#suffix' => '</div>',
          ];
        }
        else {
          foreach ($remoteFilters as $filter) {
            $form['settings']['filter_mappings'][$filter['slug']] = [
              '#type' => 'select',
              '#title' => $this->t(
                'Filter: @label (@slug)',
                [
                  '@label' => $filter['label'] ?? $filter['slug'] ?? '',
                  '@slug' => $filter['slug'],
                ]
              ),
              '#description' => $this->t(
                'Select the Drupal field to filter by for this Mapsemble filter.'
              ),
              '#options' => $fieldOptions,
              '#empty_option' => $this->t('- Do not map -'),
              '#default_value' => $this->entity->get(
                  'settings'
              )['filter_mappings'][$filter['slug']] ?? '',
            ];
          }
        }
      }
    }

    // Field mappings: Mapsemble fields to Drupal fields.
    $form['settings']['field_mappings'] = [
      '#type' => 'details',
      '#title' => $this->t('Field mappings'),
      '#description' => $this->t(
        'Map Mapsemble fields to Drupal fields for syncing.'
      ),
      '#open' => TRUE,
      '#tree' => TRUE,
      '#access' => ($mapId !== '__create__'),
    ];

    // Discover Mapsemble fields from the selected remote map (agnostic by type).
    $remoteFields = [];
    if (!empty($this->selectedRemoteMap['layers'])) {
      // Collect remote field slugs from layers to exclude from syncing.
      $remoteFieldSlugs = [];
      foreach ($this->selectedRemoteMap['layers'] as $layerInfo) {
        $rf = $layerInfo['dataSource']['config']['remoteField'] ?? NULL;
        if (!empty($rf)) {
          $remoteFieldSlugs[] = $rf;
        }
      }

      // Helper to register a field with slug, type and label.
      $addRemoteField = static function (array &$acc, ?string $slug, ?string $type, ?string $label) use ($remoteFieldSlugs): void {
        if (!$slug) {
          return;
        }
        if (in_array($slug, $remoteFieldSlugs, TRUE)) {
          // Skip the remote id field from syncing.
          return;
        }
        if (!$type) {
          // If type is unknown, we can still keep it; plugins may or may not match.
          $type = '';
        }
        // First one wins; layers could repeat slugs.
        if (!isset($acc[$slug])) {
          $acc[$slug] = [
            'type' => $type,
            'label' => $label ?: $slug,
          ];
        }
      };

      foreach ($this->selectedRemoteMap['layers'] as $layer) {
        // Newer format: dataSource.config.fields[]
        if (!empty($layer['dataSource']['config']['fields']) && is_array($layer['dataSource']['config']['fields'])) {
          foreach ($layer['dataSource']['config']['fields'] as $field) {
            $type = $field['type'] ?? '';
            $slug = $field['slug'] ?? ($field['name'] ?? NULL);
            $label = $field['label'] ?? $slug;
            $addRemoteField($remoteFields, $slug, $type, $label);
          }
        }

        // Legacy fallback: layer.config.fields[]
        if (!empty($layer['config']['fields']) && is_array($layer['config']['fields'])) {
          foreach ($layer['config']['fields'] as $field) {
            $type = $field['type'] ?? '';
            $slug = $field['slug'] ?? ($field['name'] ?? NULL);
            $label = $field['label'] ?? $slug;
            $addRemoteField($remoteFields, $slug, $type, $label);
          }
        }

        // Properties in dataSource.config.properties
        if (!empty($layer['dataSource']['config']['properties']) && is_array($layer['dataSource']['config']['properties'])) {
          foreach ($layer['dataSource']['config']['properties'] as $slug => $property) {
            $type = $property['type'] ?? '';
            $label = $property['title'] ?? $slug;
            $addRemoteField($remoteFields, $slug, $type, $label);
          }
        }

        // Legacy fallback: layer.config.properties
        if (!empty($layer['config']['properties']) && is_array($layer['config']['properties'])) {
          foreach ($layer['config']['properties'] as $slug => $property) {
            $type = $property['type'] ?? '';
            $label = $property['title'] ?? $slug;
            $addRemoteField($remoteFields, $slug, $type, $label);
          }
        }
      }
    }

    // Build Drupal field options per Mapsemble field, driven by Field plugins.
    $fieldOptionsBySlug = [];

    // Index Drupal bundle field definitions once.
    $bundleFields = $allFields ?? [];

    foreach ($remoteFields as $slug => $info) {
      $remoteType = (string) ($info['type'] ?? '');
      $options = [];

      // For each Drupal field on the selected bundle, check if any Field plugin
      // claims compatibility between this Drupal field type and the Mapsemble field type.
      foreach ($bundleFields as $fieldName => $fieldDefinition) {
        $drupalType = $fieldDefinition->getType();
        $compatible = $this->fieldPluginManager->getCompatibleDefinitions($drupalType, $remoteType);
        if (!empty($compatible)) {
          $options[$fieldName] = $fieldDefinition->getLabel();
        }
      }

      $fieldOptionsBySlug[$slug] = $options;
    }

    if (empty($remoteFields)) {
      $form['settings']['field_mappings']['_no_remote_fields'] = [
        '#type' => 'item',
        '#markup' => $this->t(
          'No Mapsemble fields were detected in the selected map. You can configure fields in Mapsemble and return to map them here.'
        ),
        '#prefix' => '<div class="messages messages--warning">',
        '#suffix' => '</div>',
      ];
    }
    else {
      foreach ($remoteFields as $slug => $info) {
        $label = $info['label'] ?? $slug;
        $remoteType = (string) ($info['type'] ?? '');

        $form['settings']['field_mappings'][$slug] = [
          '#type' => 'select',
          '#title' => $this->t('Mapsemble field: @label (@slug) [type: @type]', [
            '@label' => $label,
            '@slug' => $slug,
            '@type' => $remoteType ?: $this->t('unknown'),
          ]),
          '#description' => $this->t('Select the Drupal field to map to this Mapsemble field.'),
          '#options' => $fieldOptionsBySlug[$slug] ?? [],
          '#empty_option' => $this->t('- Do not map -'),
          '#default_value' => $this->entity->get('settings')['field_mappings'][$slug] ?? '',
          '#ajax' => [
            'callback' => '::refreshEntityConfig',
            'wrapper' => 'mapsemble-map-form-wrapper',
          ],
        ];

        // When a Drupal field is selected, show a formatter selector and its
        // settings for that field, rendered directly below the mapping select.
        $selected_field_name = $form_state->getValue(
          ['settings', 'field_mappings', $slug]
        )
          ?? $this->entity->get('settings')['field_mappings'][$slug] ?? '';
        if (!empty($selected_field_name) && isset(($allFields ?? [])[$selected_field_name])) {
          // Determine applicable formatters by field type.
          $field_def = $allFields[$selected_field_name];
          $field_type = $field_def->getType();
          $definitions = $this->formatterPluginManager->getDefinitions();
          $formatter_options = [];
          foreach ($definitions as $plugin_id => $definition) {
            $fts = $definition['field_types'] ?? [];
            if (in_array($field_type, $fts, TRUE)) {
              $label_text = (string) ($definition['label'] ?? $plugin_id);
              $formatter_options[$plugin_id] = $label_text;
            }
          }

          // Container to hold the formatter widgets but store values under a
          // different parents tree to avoid interfering with field_mappings
          // validation. This container lives visually here, but saves to
          // settings[field_formatters][slug][...].
          $form['settings']['field_mappings']['_formatter_' . $slug] = [
            '#type' => 'container',
            '#attributes' => ['class' => ['mapsemble-field-formatter']],
          ];

          $saved_formatter = $form_state->getValue(
            ['settings', 'field_formatters', $slug, 'plugin']
          )
            ?? $this->entity->get(
            'settings'
          )['field_formatters'][$slug]['plugin'] ?? '';

          $form['settings']['field_mappings']['_formatter_' . $slug]['plugin'] = [
            '#type' => 'select',
            '#title' => $this->t('Formatter'),
            '#options' => $formatter_options,
            '#empty_option' => $this->t('- Select a formatter -'),
            '#default_value' => $saved_formatter,
            '#ajax' => [
              'callback' => '::refreshEntityConfig',
              'wrapper' => 'mapsemble-map-form-wrapper',
            ],
            // Store under settings[field_formatters][slug][plugin].
            '#parents' => ['settings', 'field_formatters', $slug, 'plugin'],
          ];

          // Render formatter settings when a formatter has been chosen.
          if (!empty($saved_formatter) && isset($definitions[$saved_formatter])) {
            $saved_settings = $form_state->getValue(
              ['settings', 'field_formatters', $slug, 'settings']
            )
              ?? $this->entity->get(
              'settings'
            )['field_formatters'][$slug]['settings'] ?? [];

            // Instantiate the formatter with the current settings.
            $formatter = $this->formatterPluginManager->createInstance(
              $saved_formatter,
              [
                'field_definition' => $field_def,
                'settings' => $saved_settings,
                'view_mode' => 'full',
                'third_party_settings' => [],
                'label' => '',
              ]
            );

            $settings_form = $formatter->settingsForm([], $form_state);
            // Ensure the settings subtree is properly namespaced and tree.
            $settings_form['#parents'] = [
              'settings',
              'field_formatters',
              $slug,
              'settings',
            ];
            $settings_form['#tree'] = TRUE;

            $form['settings']['field_mappings']['_formatter_' . $slug]['settings'] = $settings_form;
          }
        }
      }
    }

    if (($bundle ?? NULL) && empty($fields)) {
      $this->messenger()->addWarning(
        $this->t(
          'No geofield found for the selected entity type and bundle. Please select a different entity.'
        )
      );
    }

    $form['settings']['location_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Location field'),
      '#default_value' => $this->entity->get(
          'settings'
      )['location_field'] ?? '',
      '#ajax' => [
        'callback' => '::refreshEntityConfig',
        'wrapper' => 'mapsemble-map-form-wrapper',
      ],
      '#options' => array_map(
        fn($field) => $field->label(),
        $fields ?? []
      ),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#access' => ((bool) ($bundle ?? NULL)) && ($mapId !== '__create__'),
    ];

    $location_field = $form_state->getValue([
      'settings',
      'location_field',
    ]) ?? $this->entity->get('settings')['location_field'] ?? '';

    $form['settings']['popup_view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Popup view mode'),
      '#description' => $this->t(
        'The view mode to use for the popup in Mapsemble.'
      ),
      '#default_value' => $this->entity->get(
          'settings'
      )['popup_view_mode'] ?? '',
      '#options' => array_map(
        fn($view_mode) => $view_mode['label'],
        $viewModes ?? [],
      ),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#access' => ((bool) $location_field) && ($mapId !== '__create__'),
    ];
    $form['settings']['card_view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Card view mode'),
      '#description' => $this->t(
        'The view mode to use for the card view in Mapsemble.'
      ),
      '#default_value' => $this->entity->get(
          'settings'
      )['card_view_mode'] ?? '',
      '#options' => array_map(
        fn($view_mode) => $view_mode['label'],
        $viewModes ?? []
      ),
      '#empty_option' => $this->t('- Select -'),
      '#required' => TRUE,
      '#access' => ((bool) $location_field) && ($mapId !== '__create__'),
    ];

    return $form;
  }

  /**
   * Gets the entity type options.
   *
   * @return array
   *   The entity type options.
   */
  protected function getEntityTypeOptions() {
    $options = [];
    foreach (
      $this->entityTypeManager->getDefinitions(
      ) as $entity_type_id => $entity_type
    ) {
      if (
        $entity_type->entityClassImplements(
          ContentEntityInterface::class
        )
      ) {
        $options[$entity_type_id] = $entity_type->getLabel();
      }
    }

    return $options;
  }

  /**
   * Reloads the configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated form array.
   */
  public function refreshEntityConfig(
    array $form,
    FormStateInterface $form_state,
  ) {
    // Return the full form so the outer wrapper refreshes actions and settings.
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    parent::validateForm($form, $form_state);

    // Skip validation when clicking the AJAX "Create" button.
    $trigger = $form_state->getTriggeringElement();
    if (!empty($trigger['#array_parents'])) {
      $last = end($trigger['#array_parents']);
      if ($last === 'create_map') {
        return;
      }
    }

    $remote_slug = $form_state->getValue(['settings', 'mapsemble', 'remoteSlug']
    )
      ?? $form_state->getValue(['mapsemble', 'remoteIdSlug'])
      ?? NULL;

    if (empty($remote_slug)) {
      // Attach error to the map selector as remoteSlug is derived from it.
      $form_state->setErrorByName(
        'settings][mapsemble][mapId',
        $this->t(
          'The selected Mapsemble map does not have its remote field configured. Please configure the remote field on the map in Mapsemble before saving.'
        )
      );
    }

    // Validate that mapped filters apply to the selected field types.
    $mappings = $form_state->getValue(['settings', 'filter_mappings']) ?? [];
    if (!empty($mappings)) {
      $entity_type_id = $form_state->getValue(['settings', 'entity_type_id']
      ) ?? $this->entity->get('settings')['entity_type_id'] ?? '';
      $bundle = $form_state->getValue(['settings', 'bundle']
      ) ?? $this->entity->get('settings')['bundle'] ?? '';
      if ($entity_type_id && $bundle) {
        $bundleFields = $this->entityFieldManager->getFieldDefinitions(
          $entity_type_id,
          $bundle
        );
        foreach ($mappings as $slug => $fieldName) {
          if (!$fieldName) {
            continue;
          }
          if (!isset($bundleFields[$fieldName])) {
            $form_state->setErrorByName(
              "settings][filter_mappings][$slug",
              $this->t('The selected field does not exist.')
            );
            continue;
          }
          $fieldType = $bundleFields[$fieldName]->getType();
          if (!$this->filterPluginManager->getPluginForFieldType($fieldType)) {
            $form_state->setErrorByName(
              "settings][filter_mappings][$slug",
              $this->t(
                'The selected field type (@type) is not supported for filtering.',
                ['@type' => $fieldType]
              )
            );
          }
        }
      }
    }

    // Validate field mappings against Field plugins using actual Mapsemble field types.
    $field_mappings = $form_state->getValue(['settings', 'field_mappings']) ?? [];
    if (!empty($field_mappings)) {
      $entity_type_id = $form_state->getValue(['settings', 'entity_type_id'])
        ?? $this->entity->get('settings')['entity_type_id'] ?? '';
      $bundle = $form_state->getValue(['settings', 'bundle'])
        ?? $this->entity->get('settings')['bundle'] ?? '';
      if ($entity_type_id && $bundle) {
        $bundleFields = $this->entityFieldManager->getFieldDefinitions(
          $entity_type_id,
          $bundle
        );

        // Build a quick lookup of Mapsemble field types by slug for validation.
        $remoteFieldTypes = [];
        if (!empty($this->selectedRemoteMap['layers'])) {
          // Collect remote field slugs to skip (remote id).
          $remoteFieldSlugs = [];
          foreach ($this->selectedRemoteMap['layers'] as $layerInfo) {
            $rf = $layerInfo['dataSource']['config']['remoteField'] ?? NULL;
            if (!empty($rf)) {
              $remoteFieldSlugs[] = $rf;
            }
          }

          $register = static function (array &$acc, ?string $slug, ?string $type) use ($remoteFieldSlugs): void {
            if (!$slug || in_array($slug, $remoteFieldSlugs, TRUE)) {
              return;
            }
            if (!isset($acc[$slug])) {
              $acc[$slug] = (string) ($type ?? '');
            }
          };

          foreach ($this->selectedRemoteMap['layers'] as $layer) {
            if (!empty($layer['dataSource']['config']['fields']) && is_array($layer['dataSource']['config']['fields'])) {
              foreach ($layer['dataSource']['config']['fields'] as $field) {
                $register($remoteFieldTypes, $field['slug'] ?? ($field['name'] ?? NULL), $field['type'] ?? '');
              }
            }
            if (!empty($layer['config']['fields']) && is_array($layer['config']['fields'])) {
              foreach ($layer['config']['fields'] as $field) {
                $register($remoteFieldTypes, $field['slug'] ?? ($field['name'] ?? NULL), $field['type'] ?? '');
              }
            }
            if (!empty($layer['dataSource']['config']['properties']) && is_array($layer['dataSource']['config']['properties'])) {
              foreach ($layer['dataSource']['config']['properties'] as $slug => $property) {
                $register($remoteFieldTypes, $slug, $property['type'] ?? '');
              }
            }
            if (!empty($layer['config']['properties']) && is_array($layer['config']['properties'])) {
              foreach ($layer['config']['properties'] as $slug => $property) {
                $register($remoteFieldTypes, $slug, $property['type'] ?? '');
              }
            }
          }
        }

        foreach ($field_mappings as $slug => $fieldName) {
          // Skip unmapped placeholders and internal keys.
          if (!$fieldName || str_starts_with((string) $slug, '_')) {
            continue;
          }
          // Do not validate mappings for the remote field; it must not be synced.
          $remote_slug = $form_state->getValue(['settings', 'mapsemble', 'remoteSlug'])
            ?? $form_state->getValue(['mapsemble', 'remoteIdSlug'])
            ?? NULL;
          if (!empty($remote_slug) && $slug === $remote_slug) {
            continue;
          }
          if (!isset($bundleFields[$fieldName])) {
            $form_state->setErrorByName(
              "settings][field_mappings][$slug",
              $this->t('The selected field does not exist.')
            );
            continue;
          }
          $fieldType = $bundleFields[$fieldName]->getType();
          $mapsembleType = $remoteFieldTypes[$slug] ?? '';

          $compatible = $this->fieldPluginManager->getCompatibleDefinitions(
            $fieldType,
            $mapsembleType
          );
          if (empty($compatible)) {
            $form_state->setErrorByName(
              "settings][field_mappings][$slug",
              $this->t(
                'No mapping plugin exists for Drupal field type (@type) to Mapsemble field type (@mtype).',
                ['@type' => $fieldType, '@mtype' => $mapsembleType ?: $this->t('unknown')]
              )
            );
          }

          // Validate selected formatter for this mapped field, if any.
          $formatter_conf = $form_state->getValue(['settings', 'field_formatters', $slug]) ?? [];
          $formatter_id = $formatter_conf['plugin'] ?? '';
          if ($formatter_id) {
            $definitions = $this->formatterPluginManager->getDefinitions();
            if (!isset($definitions[$formatter_id])) {
              $form_state->setErrorByName(
                "settings][field_formatters][$slug][plugin",
                $this->t('The selected formatter is not available.')
              );
            }
            else {
              $allowed_types = $definitions[$formatter_id]['field_types'] ?? [];
              if (!in_array($fieldType, $allowed_types, TRUE)) {
                $form_state->setErrorByName(
                  "settings][field_formatters][$slug][plugin",
                  $this->t(
                    'The selected formatter is not compatible with the field type (@type).',
                    ['@type' => $fieldType]
                  )
                );
              }
            }
          }
        }
      }
    }
  }

  /**
   * AJAX submit handler to create a new map via Mapsemble API.
   */
  public function createNewMap(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    $selected = $form_state->getValue(['settings', 'mapsemble', 'mapId']);
    if ($selected !== '__create__') {
      // Nothing to do.
      return;
    }

    $name = trim((string) ($form_state->getValue('label') ?? ''));
    if ($name === '') {
      $form_state->setErrorByName(
        'label',
        $this->t('Please enter a label before creating a new map.')
      );
      $form_state->setRebuild(TRUE);

      return;
    }

    try {
      $data = $this->mapsembleApi->createDrupalMap($name, 'drupal.vanilla');
    }
    catch (\Throwable $e) {
      $this->messenger()->addError(
        $this->t('Error creating map: @msg', ['@msg' => $e->getMessage()])
      );
      $form_state->setRebuild(TRUE);

      return;
    }

    $newId = $data['id'] ?? NULL;
    if (!$newId) {
      $this->messenger()->addError(
        $this->t('Mapsemble did not return a new map ID.')
      );
      $form_state->setRebuild(TRUE);

      return;
    }

    // Set the selected Map ID to the newly created one.
    $form_state->setValue(['settings', 'mapsemble', 'mapId'], $newId);
    // Also update the raw user input so the rebuilt form reflects the new selection
    // and departs from the previous '__create__' value used to reveal the button.
    $input = $form_state->getUserInput();
    if (is_array($input)) {
      $input['settings']['mapsemble']['mapId'] = $newId;
      $form_state->setUserInput($input);
    }

    $this->messenger()->addStatus(
      $this->t(
        'Map "@label" has been created and selected.',
        ['@label' => $data['label'] ?? $name]
      )
    );

    // Rebuild the form to load fresh remote maps and dependent settings.
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    // Sanitize settings before saving: remove any placeholder keys (e.g., _no_remote_filters)
    // that may have been added to the form for UX purposes only.
    $settings = $this->entity->get('settings') ?? [];
    foreach (['filter_mappings', 'field_mappings'] as $mapKey) {
      if (!empty($settings[$mapKey]) && is_array($settings[$mapKey])) {
        foreach (array_keys($settings[$mapKey]) as $k) {
          if (is_string($k) && str_starts_with($k, '_')) {
            unset($settings[$mapKey][$k]);
          }
        }
      }
    }
    $this->entity->set('settings', $settings);
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match ($result) {
        \SAVED_NEW => $this->t('Created new example %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated example %label.', $message_args),
      }
    );
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));

    return $result;
  }

}
