<?php

declare(strict_types=1);

namespace Drupal\mapsemble\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Formatter that renders the selected Mapsemble map using the render element.
 */
#[FieldFormatter(
  id: 'mapsemble_map',
  label: new \Drupal\Core\StringTranslation\TranslatableMarkup('Mapsemble map'),
  field_types: ['mapsemble_map']
)]
final class MapsembleMapFormatter extends FormatterBase implements ContainerFactoryPluginInterface {

  public function __construct(
    string $plugin_id,
    mixed $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'height' => '50vh',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = [];
    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Map height (CSS value)'),
      '#default_value' => $this->getSetting('height'),
      '#description' => $this->t('Any valid CSS height value such as 400px or 50vh.'),
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    return [
      $this->t('Height: @height', ['@height' => $this->getSetting('height')]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];

    $storage = $this->entityTypeManager->getStorage('mapsemble_map');

    foreach ($items as $delta => $item) {
      $map_id = (string) $item->get('map_id')->getString();
      if ($map_id === '') {
        continue;
      }
      $map = $storage->load($map_id);
      if (!$map) {
        continue;
      }
      $elements[$delta] = [
        '#type' => 'mapsemble_map',
        '#map' => $map,
        '#height' => (string) $this->getSetting('height'),
      ];
    }

    return $elements;
  }

}
