<?php

namespace Drupal\markdown\Plugin\Markdown\CommonMark\Extension;

use League\CommonMark\Extension\HeadingPermalink\HeadingPermalinkRenderer;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Theme\ActiveTheme;
use Drupal\markdown\Plugin\Markdown\AllowedHtmlInterface;
use Drupal\markdown\Plugin\Markdown\CommonMark\BaseExtension;
use Drupal\markdown\Plugin\Markdown\ParserInterface;
use Drupal\markdown\Plugin\Markdown\SettingsInterface;
use Drupal\markdown\Traits\SettingsTrait;

/**
 * Integrates the Heading Permalink extension with configurable settings.
 *
 * @MarkdownAllowedHtml(
 *   id = "commonmark-heading-permalink",
 * )
 * @MarkdownExtension(
 *   id = "commonmark-heading-permalink",
 *   label = @Translation("Heading Permalink"),
 *   description = @Translation("Makes all heading elements (&lt;h1&gt;, &lt;h2&gt;, etc) linkable so users can quickly grab a link to that specific part of the document."),
 *   libraries = {
 *     @ComposerPackage(
 *       id = "league/commonmark",
 *       object = "\League\CommonMark\Extension\HeadingPermalink\HeadingPermalinkExtension",
 *       customLabel = "commonmark-heading-permalink",
 *       url = "https://commonmark.thephpleague.com/extensions/heading-permalinks/",
 *       requirements = {
 *          @InstallableRequirement(
 *             id = "parser:commonmark",
 *             callback = "::getVersion",
 *             constraints = {"Version" = "^1.3 || ^2.0"},
 *          ),
 *       },
 *     ),
 *   },
 * )
 */
class HeadingPermalinkExtension extends BaseExtension implements AllowedHtmlInterface, PluginFormInterface, SettingsInterface {

  use SettingsTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings($pluginDefinition) {
    /** @var \Drupal\markdown\Annotation\InstallablePlugin $pluginDefinition */

    $symbol = defined('\\League\\CommonMark\\Extension\\HeadingPermalink\\HeadingPermalinkRenderer::DEFAULT_SYMBOL')
      ? HeadingPermalinkRenderer::DEFAULT_SYMBOL
      : '¶';

    return [
      'html_class' => 'heading-permalink',
      'id_prefix' => 'user-content',
      'insert' => 'before',
      'title' => 'Permalink',
      'symbol' => $symbol,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function allowedHtmlTags(ParserInterface $parser, ?ActiveTheme $activeTheme = NULL) {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $element, FormStateInterface $form_state) {
    /** @var \Drupal\markdown\Form\SubformStateInterface $form_state */

    $element += $this->createSettingElement('html_class', [
      '#type' => 'textfield',
      '#title' => $this->t('HTML Class'),
      '#description' => $this->t("The value of this nested configuration option should be a <code>string</code> that you want set as the <code>&lt;a&gt;</code> tag's class attribute."),
    ], $form_state);

    $element += $this->createSettingElement('id_prefix', [
      '#type' => 'textfield',
      '#title' => $this->t('ID Prefix'),
      '#description' => $this->t("This should be a <code>string</code> you want prepended to HTML IDs. This prevents generating HTML ID attributes which might conflict with others in your stylesheet. A dash separator (-) will be added between the prefix and the ID. You can instead set this to an empty string ('') if you don’t want a prefix."),
    ], $form_state);

    $element += $this->createSettingElement('symbol', [
      '#type' => 'textfield',
      '#title' => $this->t('Symbol'),
      '#description' => $this->t("This controls the symbol used to display the permalink. HTML is not allowed here; provide a short piece of text (for example, '¶') and style it with CSS if desired."),
    ], $form_state);

    $element += $this->createSettingElement('insert', [
      '#type' => 'select',
      '#title' => $this->t('Insert'),
      '#description' => $this->t("This controls whether the anchor is added to the beginning of the <code>&lt;h1&gt;</code>, <code>&lt;h2&gt;</code> etc. tag or to the end."),
      '#options' => [
        'after' => $this->t('After'),
        'before' => $this->t('Before'),
      ],
    ], $form_state);

    $element += $this->createSettingElement('title', [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#description' => $this->t("This option sets the title attribute on the <code>&lt;a&gt;</code> tag."),
    ], $form_state);

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsKey() {
    return 'heading_permalink';
  }

}
