<?php

namespace Drupal\markdown\Plugin\migrate\process;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\markdown\PluginManager\ParserManagerInterface;
use Drupal\migrate\ProcessPluginBase;
use Drupal\migrate\MigrateExecutableInterface;
use Drupal\migrate\Row;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Generates Markdown settings during migration.
 *
 * @MigrateProcessPlugin(
 *   id = "markdown_settings_generator",
 *   handle_multiples = TRUE
 * )
 */
class MarkdownSettingsGenerator extends ProcessPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The markdown parser plugin manager.
   *
   * @var \Drupal\markdown\PluginManager\ParserManagerInterface
   */
  protected $parserManager;

  /**
   * Constructs a MarkdownSettingsGenerator process plugin.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\markdown\PluginManager\ParserManagerInterface $parserManager
   *   The markdown parser plugin manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ParserManagerInterface $parserManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->parserManager = $parserManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.markdown.parser')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function transform($value, MigrateExecutableInterface $migrate_executable, Row $row, $destination_property) {
    if ($row->getDestinationProperty('id') === 'markdown') {
      // The markdown module during installation will automatically detect the
      // available parsers and select a default.
      // @see markdown_requirements()
      // @todo Add support for generating valid configuration for other parsers?
      if ($this->parserManager->getDefaultParser()->getPluginId() === 'commonmark') {
        // @see markdown_migration_plugins_alter()
        $source_format = $row->getSourceProperty('source_format');
        $enabled_filters = array_column($source_format['filters'], 'name');
        $value = [
          'id' => 'commonmark',
          'enabled' => TRUE,
          'render_strategy' => [
            'type' => 'filter_output',
            'custom_allowed_html' => '',
            'plugins' => [
              'commonmark' => TRUE,
              // Footnotes were supported by default in the D7 version.
              'commonmark-footnotes' => TRUE,
              // Enable Markdown support for specific filters if the filters are
              // enabled.
              'filter_align' => in_array('filter_align', $enabled_filters),
              'filter_caption' => in_array('filter_caption', $enabled_filters),
              'media_embed' => in_array('filter_caption', $enabled_filters),
            ],
          ],
          'settings' => [
            'html_input' => 'allow',
            'max_nesting_level' => 0,
          ],
          'extensions' => [
            // Footnotes were supported by default in the D7 version.
            [
              'id' => 'commonmark-footnotes',
              'enabled' => TRUE,
              'weight' => 0,
              'settings' => [
                'container_add_hr' => TRUE,
              ],
            ],
          ],
          'override' => TRUE,
        ];
      }
    }

    return $value;
  }

}
