<?php

namespace Drupal\Tests\markdown\Functional;

use Drupal\editor\EditorInterface;
use Drupal\editor\Entity\Editor;
use Drupal\filter\Entity\FilterFormat;
use Drupal\filter\FilterFormatInterface;
use Drupal\markdown\Plugin\Filter\FilterMarkdownInterface;

/**
 * Makes sure that the markdown filter can be enabled for a text format.
 *
 * @group markdown
 */
class FilterAdminTest extends MarkdownBrowserTestBase {

  /**
   * A user with administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer filters',
      'access site reports',
    ]);

    $this->drupalLogin($this->adminUser);
  }

  /**
   * Tests creating a text format with markdown filter enabled.
   */
  public function testEnableMarkdownFilter() {
    // Add text format.
    $this->drupalGet('admin/config/content/formats/add');

    $format_id = $this->randomMachineName();
    $name = $this->randomMachineName();
    $edit = [
      'format' => $format_id,
      'name' => $name,
      'filters[markdown][status]' => 1,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Check that the filter has been created.
    $format = FilterFormat::load($format_id);
    $this->assertInstanceof(FilterFormatInterface::class, $format);

    // And check that markdown filter is enabled.
    $markdown = $format->filters()->get('markdown');
    $this->assertInstanceof(FilterMarkdownInterface::class, $markdown);
    $this->assertTrue($markdown->isEnabled());
  }

  /**
   * Tests creating a text format using CKEditor without the markdown filter.
   */
  public function testCreateTextFormatWithoutMarkdownFilter() {
    // Enable the CKEditor5 module.
    $this->container->get('module_installer')->install(['ckeditor5']);

    // Add text format.
    $this->drupalGet('admin/config/content/formats/add');

    $format_id = $this->randomMachineName();
    $edit = [
      'format' => $format_id,
      'name' => $this->randomMachineName(),
      'editor[editor]' => 'ckeditor5',
    ];
    $this->submitForm($edit, 'Configure');

    // Workaround for https://www.drupal.org/project/drupal/issues/3457717.
    $edit = [
      'editor[settings][toolbar][items]' => '["heading","bold","italic","drupalInsertImage"]',
    ];
    $this->submitForm($edit, 'Configure');
    $this->submitForm([], 'Save configuration');
    $this->assertSession()->statusCodeEquals(200);

    // Check that the filter has been created.
    $format = FilterFormat::load($format_id);
    $this->assertInstanceof(FilterFormatInterface::class, $format);

    // Check that CKEditor5 is enabled for this format.
    $editor = Editor::load($format_id);
    $this->assertInstanceof(EditorInterface::class, $editor);
    $this->assertEquals('ckeditor5', $editor->getEditor());
  }

  /**
   * Tests enabling markdown when there are no parsers available.
   */
  public function testEnableMarkdownFilterWithoutParsers() {
    // Simulate the situation of no parsers being available.
    $this->container->get('module_installer')->install(['markdown_test_no_parsers']);

    // Add text format.
    $this->drupalGet('admin/config/content/formats/add');

    $format_id = $this->randomMachineName();
    $name = $this->randomMachineName();
    $edit = [
      'format' => $format_id,
      'name' => $name,
      'filters[markdown][status]' => 1,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Check that the filter has been created.
    $format = FilterFormat::load($format_id);
    $this->assertInstanceof(FilterFormatInterface::class, $format);

    // And check that markdown filter is enabled.
    $markdown = $format->filters()->get('markdown');
    $this->assertInstanceof(FilterMarkdownInterface::class, $markdown);
    $this->assertTrue($markdown->isEnabled());
  }

}
