<?php

namespace Drupal\Tests\markdown\Functional;

use Drupal\Component\Utility\NestedArray;
use Drupal\Tests\BrowserTestBase;
use Drupal\filter\Entity\FilterFormat;

/**
 * Provides a base class for Markdown functional tests.
 */
abstract class MarkdownBrowserTestBase extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'filter',
    'markdown',
    'system',
  ];

  /**
   * Creates or updates a Markdown text format with the given settings.
   *
   * @param array $markdown_settings
   *   (optional) An array of settings for the markdown filter.
   * @param string $format_id
   *   (optional) The machine name of the text format. Defaults to 'markdown'.
   *
   * @return \Drupal\filter\Entity\FilterFormat
   *   The created or updated text format entity.
   */
  protected function createMarkdownFilterFormat(array $markdown_settings = [], string $format_id = 'markdown'): FilterFormat {
    // Remove existing format if it exists, to ensure a clean state.
    if ($existing = FilterFormat::load($format_id)) {
      $existing->delete();
    }

    $markdown_default_settings = [
      'id' => 'commonmark',
      'enabled' => TRUE,
      'render_strategy' => [
        'type' => 'filter_output',
        'custom_allowed_html' => '',
        'plugins' => [
          'commonmark' => TRUE,
        ],
      ],
    ];
    $markdown_settings = NestedArray::mergeDeep($markdown_default_settings, $markdown_settings);

    $format = FilterFormat::create([
      'format' => $format_id,
      'name' => ucfirst($format_id),
      'filters' => [
        'markdown' => [
          'id' => 'markdown',
          'provider' => 'markdown',
          'status' => TRUE,
          'settings' => $markdown_settings,
        ],
      ],
    ]);
    $format->save();

    return $format;
  }

  /**
   * Asserts that the rendered table contains the expected cell values.
   *
   * @param string[][] $expected
   *   A 2D array representing the expected table contents by rows.
   */
  protected function assertRenderedTableEquals(array $expected): void {
    $rows = $this->getSession()->getPage()->findAll('css', 'table tr');
    $actual = [];

    foreach ($rows as $row) {
      $cells = $row->findAll('css', 'th, td');
      $actual_row = [];
      foreach ($cells as $cell) {
        $actual_row[] = $cell->getText();
      }
      $actual[] = $actual_row;
    }

    $this->assertEquals($expected, $actual);
  }

}
