<?php

namespace Drupal\Tests\markdown\Functional\Plugin\Markdown;

/**
 * Tests the CommonMark parser.
 *
 * @group markdown
 */
class CommonMarkTest extends MarkdownParserTestBase {

  /**
   * {@inheritdoc}
   */
  protected function getPluginId(): string {
    return 'commonmark';
  }

  /**
   * Tests enable the external links extension.
   */
  public function testEnableExternalLinks(): void {
    $this->loginAsMarkdownAdmin();
    $this->drupalGet($this->getConfigurePath());

    $edit = [
      'extensions[commonmark-external-links][enabled]' => 1,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Check that the external links extension has been enabled.
    $config = $this->container->get('config.factory')
      ->get('markdown.parser.commonmark');

    // Find the index number for the 'commonmark-external-links' extension.
    $extensions_config = $config->get('extensions');
    $ids = array_column($extensions_config, 'id');
    $key = array_search('commonmark-external-links', $ids);

    $this->assertTrue($extensions_config[$key]['enabled']);
  }

  /**
   * Tests that a Markdown table is rendered properly.
   */
  public function testMarkdownTableRendering(): void {
    $this->drupalCreateContentType([
      'type' => 'page',
    ]);

    // Enable the 'table' extension for the CommonMark parser.
    $this->container->get('config.factory')
      ->getEditable('markdown.parser.commonmark')
      ->set('extensions', [
        [
          'id' => 'commonmark-table',
          'enabled' => TRUE,
          'weight' => 0,
        ],
      ])
      ->save();

    // Create a text format with markdown filter and table rendering enabled.
    $format = $this->createMarkdownFilterFormat([
      'render_strategy' => [
        'plugins' => [
          'commonmark-table' => TRUE,
        ],
      ],
    ]);

    // Create a user who may create content with markdown.
    $markdown_user = $this->drupalCreateUser([
      'create page content',
      'edit own page content',
      'use text format ' . $format->id(),
    ]);

    $this->drupalLogin($markdown_user);

    $markdown = <<<EOD
| Header 1 | Header 2 |
|----------|----------|
| Cell 1   | Cell 2   |
| Cell 3   | Cell 4   |
EOD;

    $this->drupalGet('node/add/page');
    $edit = [
      'title[0][value]' => 'Markdown Table Test',
      'body[0][value]' => $markdown,
    ];
    $this->submitForm($edit, 'Save');

    // Check if the table is rendered correctly.
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->elementExists('css', 'table');
    $this->assertSession()->elementContains('css', 'th', 'Header 1');
    $this->assertRenderedTableEquals([
      ['Header 1', 'Header 2'],
      ['Cell 1', 'Cell 2'],
      ['Cell 3', 'Cell 4'],
    ]);
  }

}
