<?php

namespace Drupal\Tests\markdown\Functional\Plugin\Markdown;

use Drupal\Tests\markdown\Functional\MarkdownBrowserTestBase;
use Drupal\filter\Entity\FilterFormat;
use Drupal\filter\FilterFormatInterface;
use Drupal\markdown\Plugin\Filter\FilterMarkdownInterface;
use Drupal\user\UserInterface;

/**
 * Base class for MarkdownParser plugin tests.
 */
abstract class MarkdownParserTestBase extends MarkdownBrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'user',
    'filter',
    'text',
    'markdown',
  ];

  /**
   * Tests that the settings form can be submitted.
   */
  public function testConfigure() {
    $this->loginAsMarkdownAdmin();
    $this->drupalGet($this->getConfigurePath());
    $this->submitForm([], 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Tests creating a text format with markdown filter enabled.
   */
  public function testEnableMarkdownFilter() {
    $this->loginAsFilterAdmin();

    // Add text format.
    $this->drupalGet('admin/config/content/formats/add');

    $format_id = $this->randomMachineName();
    $name = $this->randomMachineName();
    $edit = [
      'format' => $format_id,
      'name' => $name,
      'filters[markdown][status]' => 1,
      'filters[markdown][settings][id]' => $this->getPluginId(),
    ];
    $this->submitForm($edit, 'Save configuration');

    // Check that the filter has been created.
    $format = FilterFormat::load($format_id);
    $this->assertInstanceof(FilterFormatInterface::class, $format);

    // And check that the markdown filter is enabled.
    $markdown = $format->filters()->get('markdown');
    $this->assertInstanceof(FilterMarkdownInterface::class, $markdown);
    $this->assertTrue($markdown->isEnabled());
    // And with the expected parser.
    $config = $markdown->getConfiguration();
    $this->assertEquals($this->getPluginId(), $config['settings']['id']);
    $this->assertTrue($config['settings']['enabled']);
  }

  /**
   * Tests rendering basic markdown in the user interface.
   */
  public function testMarkdownRender() {
    // Create a content type with body field.
    $this->drupalCreateContentType([
      'type' => 'page',
    ]);

    // Create a text format with markdown filter.
    $format = $this->createMarkdownFilterFormat();

    // Create a user who may create content with markdown.
    $markdown_user = $this->drupalCreateUser([
      'create page content',
      'edit own page content',
      'use text format ' . $format->id(),
    ]);

    $this->drupalLogin($markdown_user);

    // Create a node with Markdown input.
    $this->drupalGet('node/add/page');
    $edit = [
      'title[0][value]' => 'Test Markdown',
      'body[0][value]' => '**bold text**',
    ];
    $this->submitForm($edit, 'Save');

    // Confirm the output is rendered as HTML.
    $this->assertSession()->pageTextContains('Test Markdown');
    $this->assertSession()->responseContains('<strong>bold text</strong>');
  }

  /**
   * Returns the ID of the parser to test.
   *
   * @return string
   *   The parser ID to test.
   */
  abstract protected function getPluginId(): string;

  /**
   * Returns the path on where the parser can be configured.
   *
   * @return string
   *   The path where the parser can be configured.
   */
  protected function getConfigurePath(): string {
    return '/admin/config/content/markdown/' . $this->getPluginId();
  }

  /**
   * Creates an user who may administer filters and logs in as this user.
   *
   * @return \Drupal\user\UserInterface
   *   The created admin user.
   */
  protected function loginAsFilterAdmin(): UserInterface {
    $admin_user = $this->drupalCreateUser([
      'administer filters',
      'access site reports',
    ]);
    $this->drupalLogin($admin_user);

    return $admin_user;
  }

  /**
   * Creates an user who may administer markdown and logs in as this user.
   *
   * @return \Drupal\user\UserInterface
   *   The created admin user.
   */
  protected function loginAsMarkdownAdmin(): UserInterface {
    $admin_user = $this->drupalCreateUser([
      'administer markdown',
    ]);
    $this->drupalLogin($admin_user);

    return $admin_user;
  }

}
