<?php

namespace Drupal\Tests\markdown\Unit\Plugin\Filter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\ElementInfoManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Tests\markdown\Unit\MarkdownUnitTestBase;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterInterface;
use Drupal\markdown\Plugin\Filter\FilterMarkdown;
use Drupal\markdown\Plugin\Markdown\ParserInterface;
use Drupal\markdown\PluginManager\ParserManagerInterface;
use Drupal\markdown\Render\ParsedMarkdownInterface as ParsedMarkdownInterfaceBase;
use Prophecy\Argument;

/**
 * @coversDefaultClass \Drupal\markdown\Plugin\Filter\FilterMarkdown
 * @group markdown
 */
class FilterMarkdownTest extends MarkdownUnitTestBase {

  /**
   * The Element Info Manager service.
   *
   * @var \Prophecy\Prophecy\ProphecyInterface|\Drupal\Core\Render\ElementInfoManagerInterface
   */
  protected $elementInfo;

  /**
   * The Markdown Parser Plugin Manager service.
   *
   * @var \Prophecy\Prophecy\ProphecyInterface|\Drupal\markdown\PluginManager\ParserManagerInterface
   */
  protected $parserManager;

  /**
   * The Markdown Parser Plugin Manager service.
   *
   * @var \Prophecy\Prophecy\ProphecyInterface|\Drupal\markdown\Plugin\Markdown\ParserInterface
   */
  protected $parser;

  /**
   * The language manager service.
   *
   * @var \Prophecy\Prophecy\ProphecyInterface|\Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory service.
   *
   * @var \Prophecy\Prophecy\ProphecyInterface|\Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->elementInfo = $this->prophesize(ElementInfoManagerInterface::class);
    $this->parserManager = $this->prophesize(ParserManagerInterface::class);
    $this->languageManager = $this->prophesize(LanguageManagerInterface::class);
    $this->configFactory = $this->prophesize(ConfigFactoryInterface::class);

    $parsed_markdown = $this->prophesize(ParsedMarkdownInterface::class);
    $parsed_markdown->__toString()
      ->willReturn('bar');

    $this->parser = $this->prophesize(ParserInterface::class);
    $this->parser->getPluginId()
      ->willReturn('foo');
    $this->parser->parse(Argument::type('string'), NULL)
      ->willReturn($parsed_markdown->reveal());
    $this->parser->getConfiguration()
      ->willReturn([]);

    $container = new ContainerBuilder();
    $this->configFactory->get('markdown.parser.foo')
      ->willReturn($this->createMock(ImmutableConfig::class));
    $container->set('config.factory', $this->configFactory->reveal());
    \Drupal::setContainer($container);
  }

  /**
   * Returns a new FilterMarkdown instance to test with.
   *
   * @param array $configuration
   *   (optional) The configuration for the filter.
   *
   * @return \Drupal\markdown\Plugin\Filter\FilterMarkdown
   *   A FilterMarkdown instance.
   */
  protected function getInstance(array $configuration = []): FilterMarkdown {
    $plugin_definition = [
      'description' => new TranslatableMarkup('Allows content to be submitted using Markdown, a simple plain-text syntax that is filtered into valid HTML.'),
      'weight' => -15,
      'status' => TRUE,
      'settings' => [],
      'id' => 'markdown',
      'title' => new TranslatableMarkup('Markdown'),
      'type' => FilterInterface::TYPE_MARKUP_LANGUAGE,
      'class' => FilterMarkdown::class,
      'provider' => 'markdown',
    ];
    return new FilterMarkdown(
      $configuration,
      'markdown',
      $plugin_definition,
      $this->elementInfo->reveal(),
      $this->parserManager->reveal(),
      $this->languageManager->reveal(),
      $this->configFactory->reveal()
    );
  }

  /**
   * @covers ::process
   */
  public function testProcess() {
    $this->parserManager->getDefaultParser()
      ->willReturn($this->parser->reveal());
    $this->parserManager->createInstance('foo', Argument::type('array'))
      ->willReturn($this->parser->reveal());

    $filter = $this->getInstance();
    $result = $filter->process('foo');
    $this->assertInstanceof(FilterProcessResult::class, $result);
    $this->assertIsString($result->getProcessedText());
  }

}

/**
 * Extends ParsedMarkdownInterfaceBase to avoid deprecation warnings.
 *
 * ParsedMarkdownInterface extends \Serializable, this requires methods
 * __serialize() and __unserialize() to be defined. The PHP Unit mock builder
 * does not automatically defines these methods and does not think that it
 * should.
 *
 * @see https://github.com/sebastianbergmann/phpunit/issues/4783
 */
interface ParsedMarkdownInterface extends ParsedMarkdownInterfaceBase {

  /**
   * Makes sure that magic method __serialize() is defined.
   *
   * @return array
   *   An associative array of key/value pairs that represents the serialized
   *   form of the object.
   */
  public function __serialize(): array;

  /**
   * Makes sure that magic method __unserialize() is defined.
   *
   * @param array $data
   *   The restored array data that was returned from __serialize().
   */
  public function __unserialize(array $data): void;

}
