<?php

namespace Drupal\markdownify\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Adds "X-Robots-Tag: noindex" to Markdown responses.
 *
 * This subscriber ensures that Markdown representations of content (e.g.,
 * `/node/1.md`, `/markdownify/...`, `?_format=markdown`, or requests with an
 * `Accept: text/markdown` header) are not indexed by search engines.
 *
 * The detection is based on the `Content-Type` response header rather than the
 * route or request path, which guarantees that all Markdown outputs are
 * consistently covered.
 */
class MarkdownifyNoIndexSubscriber implements EventSubscriberInterface {

  /**
   * The Markdownify module configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Constructs a MarkdownifyNoIndexSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->get('markdownify.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Subscribe to the kernel RESPONSE event to inspect outgoing responses.
    return [
      KernelEvents::RESPONSE => ['onResponse', 0],
    ];
  }

  /**
   * Adds the X-Robots-Tag header to Markdown responses.
   *
   * This method runs for each main request handled by Drupal. When the response
   * content type indicates Markdown output, it adds the `X-Robots-Tag: noindex`
   * header to prevent search engines from indexing the content.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event triggered by the Symfony kernel.
   */
  public function onResponse(ResponseEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $noindex = (bool) $this->config->get('noindex');
    if (!$noindex) {
      return;
    }

    $response = $event->getResponse();
    $content_type = $response->headers->get('Content-Type');

    // Add the "noindex" header for all Markdown responses.
    if ($content_type && str_starts_with($content_type, 'text/markdown')) {
      $response->headers->set('X-Robots-Tag', 'noindex');
    }
  }

}
