<?php

namespace Drupal\markdownify\Form;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Plugin\PluginFormInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Contains configuration page for markdownify settings.
 */
class MarkdownifySettings extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The converter plugin manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected PluginManagerInterface $converterPluginManager;

  /**
   * The Supported Entities Config Form.
   *
   * @var \Drupal\Core\Plugin\PluginFormInterface
   */
  protected PluginFormInterface $supportedEntitiesConfigForm;

  /**
   * The route builder service.
   *
   * @var \Drupal\Core\Routing\RouteBuilderInterface
   */
  protected $routeBuilder;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->converterPluginManager = $container->get('plugin.manager.html_to_markdown_converter');
    $instance->supportedEntitiesConfigForm = $container->get('markdownify.supported_entities.config_form');
    $instance->routeBuilder = $container->get('router.builder');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['markdownify.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'markdownify_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('markdownify.settings');
    // Pass it to the subform through $form_state.
    $supported_entities = $config->get('supported_entities') ?? [];
    $form_state->setValue('supported_entities', $supported_entities);
    // Add the Supported Entities Config subform.
    $form += $this->supportedEntitiesConfigForm->buildConfigurationForm($form, $form_state);
    // Converters options.
    $converters = [];
    $converter_definitions = $this->converterPluginManager->getDefinitions();
    foreach ($converter_definitions as $converter_id => $converter) {
      $converters[$converter_id] = $converter['label'];
    }

    $form['noindex'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Add “noindex” to Markdown responses'),
      '#default_value' => $config->get('noindex') ?? TRUE,
      '#description' => $this->t('When enabled, Markdown responses will include the <code>X-Robots-Tag: noindex</code> HTTP header to prevent indexing by search engines.'),
    ];

    $form['default_converter'] = [
      '#type' => 'radios',
      '#title' => $this->t('Default converter'),
      '#default_value' => $config->get('default_converter'),
      '#options' => $converters,
    ];
    $conversion_settings = $config->get('converters');
    $form['converters'] = [
      '#type' => 'details',
      '#title' => $this->t('Converter settings'),
      '#tree' => TRUE,
    ];
    foreach ($converter_definitions as $converter_id => $converter) {
      $instance = $this->converterPluginManager->createInstance($converter_id, $conversion_settings[$converter_id] ?? []);
      $subform_state = SubformState::createForSubform($form['converters'], $form, $form_state);
      $form['converters'][$converter_id] = $instance->buildConfigurationForm([], $subform_state);
      $form['converters'][$converter_id] += [
        '#type' => 'details',
        '#title' => $this->t('League Html to Markdown'),
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    $this->supportedEntitiesConfigForm->validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $form_state->cleanValues();
    $values = $form_state->getValues();
    $this->config('markdownify.settings')
      ->set('default_converter', $values['default_converter'])
      ->set('converters', $values['converters'])
      ->set('supported_entities', $values['supported_entities'])
      ->set('noindex', $values['noindex'])
      ->save();
    parent::submitForm($form, $form_state);
    // Ensure router information is correctly rebuilt.
    $this->routeBuilder->setRebuildNeeded();
    Cache::invalidateTags(['rendered']);
  }

}
