<?php

namespace Drupal\markdownify\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\markdownify\MarkdownifySupportedEntityTypesValidatorInterface;
use Drupal\markdownify\Utility\MarkdownifyPath;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;

/**
 * Access checker for Markdownify routes based on entity, bundle, and language.
 *
 * This ensures that only supported entities, bundles, and languages are exposed
 * through Markdownify routes, based on the module configuration.
 *
 * Example route definition:
 *
 * @code
 * markdownify.entity:
 *   path: '/markdownify/{node}'
 *   defaults:
 *     _controller: '\Drupal\markdownify\Controller\MarkdownifyController::render'
 *   requirements:
 *     _markdownify_entity_access: 'node'
 * @endcode
 */
class MarkdownifyEntityAccessCheck implements AccessInterface {

  /**
   * Validates supported entity types, bundles, and languages for Markdownify.
   *
   * @var \Drupal\markdownify\MarkdownifySupportedEntityTypesValidatorInterface
   */
  protected MarkdownifySupportedEntityTypesValidatorInterface $validator;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a new MarkdownifyEntityAccessCheck object.
   *
   * @param \Drupal\markdownify\MarkdownifySupportedEntityTypesValidatorInterface $validator
   *   The supported entity types validator.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(MarkdownifySupportedEntityTypesValidatorInterface $validator, LanguageManagerInterface $language_manager) {
    $this->validator = $validator;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('markdownify.supported_entity_types.validator'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account): AccessResult {
    $requirement = $route->getRequirement('_markdownify_entity_access');
    $parameters = $route_match->getParameters();
    // Ensure that the route includes a valid entity parameter.
    if (!$parameters->has($requirement)) {
      $access = AccessResult::neutral("Markdownify access check skipped: route parameter '{$requirement}' not found.");
      $access->addCacheContexts(['route']);
      return $access;
    }
    $entity = $parameters->get($requirement);
    if (!$entity instanceof EntityInterface) {
      $access = AccessResult::neutral("Markdownify access check skipped: parameter '{$requirement}' is not an entity.");
      $access->addCacheContexts(['route']);
      return $access;
    }
    // Validate if entity type, bundle, and language are supported.
    $supported_entities = $this->validator->getSupportedEntities();
    $entity_type_id = $entity->getEntityTypeId();
    $configuration = $supported_entities[$entity_type_id] ?? NULL;
    if (!$configuration) {
      $access = AccessResult::forbidden("Access denied: Entity type '{$entity_type_id}' is not enabled for Markdownify.");
      $access->addCacheTags(['config:markdownify.settings']);
      return $access;
    }
    // Check bundle support.
    $supported_bundles = $configuration['bundles'] ?? [];
    $bundle = $entity->bundle();
    $bundles = MarkdownifyPath::getSupportedBundles($supported_bundles, $entity_type_id);
    if ($bundles != ['all'] && !in_array($bundle, $bundles, TRUE)) {
      $access = AccessResult::forbidden("Access denied: Bundle '{$bundle}' of entity type '{$entity_type_id}' is not enabled for Markdownify.");
      $access->addCacheTags(['config:markdownify.settings']);
      $access->addCacheableDependency($entity);
      return $access;
    }
    // Check language support.
    $language = $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT)->getId();
    $supported_languages = $configuration['languages'] ?? [];
    $languages = MarkdownifyPath::getSupportedLanguages($supported_languages);
    if ($languages != ['all'] && !in_array($language, $languages, TRUE)) {
      $access = AccessResult::forbidden("Access denied: Language '{$language}' is not enabled for Markdownify for entity type '{$entity_type_id}'.");
      $access->addCacheTags(['config:markdownify.settings']);
      $access->addCacheableDependency($entity);
      return $access;
    }
    // All checks passed.
    $access = AccessResult::allowed();
    $access->addCacheTags(['config:markdownify.settings']);
    $access->addCacheContexts(['languages', 'route']);
    $access->addCacheableDependency($entity);
    return $access;
  }

}
