<?php

/**
 * @file
 * Hooks specific to the Markdownify module.
 */

use Drupal\Core\Render\BubbleableMetadata;

/**
 * Alter the supported entity types for Markdown conversion.
 *
 * This hook allows other modules to modify the list of entity types that
 * the Markdownify module supports for conversion to Markdown.
 *
 * @param array &$supported_entity_types
 *   The list of entity types supported by the module. Default values include
 *   'node' and 'taxonomy_term'. Other modules can add or remove items from
 *   this array as needed.
 *
 * @deprecated in markdownify:1.1.1 and is removed from markdownify:2.0.0.
 * Use hook_markdownify_supported_entities_alter() instead.
 *
 * @see https://www.drupal.org/project/markdownify/issues/3547314
 */
function hook_markdownify_supported_entity_types_alter(array &$supported_entity_types): void {
  // Example: Add a custom entity type to the supported list.
  if (\Drupal::moduleHandler()->moduleExists('custom_module')) {
    $supported_entity_types[] = 'custom_entity_type';
  }
  // Example: Remove 'taxonomy_term' from the supported list.
  if (($key = array_search('taxonomy_term', $supported_entity_types)) !== FALSE) {
    unset($supported_entity_types[$key]);
  }
}

/**
 * Alter the supported entities configuration for Markdown conversion.
 *
 * This is the replacement for hook_markdownify_supported_entity_types_alter().
 *
 * The new hook provides full access to the structured configuration for
 * each supported entity type, allowing fine-grained control over which
 * bundles and languages Markdownify should process.
 *
 * @param array &$supported_entities
 *   An associative array keyed by entity type ID. Each value is an array with:
 *   - bundles: (array)
 *       - default: (bool) Whether to exclude (TRUE) or include (FALSE)
 *         the selected bundles.
 *       - selected: (string[]) List of specific bundle machine names.
 *   - languages: (array)
 *       - default: (bool) Whether to exclude (TRUE) or include (FALSE)
 *         the selected languages.
 *       - selected: (string[]) List of language codes.
 *
 * @see https://www.drupal.org/project/markdownify/issues/3547314
 */
function hook_markdownify_supported_entities_alter(array &$supported_entities): void {
  // Example: Add a new supported entity type with configuration.
  if (\Drupal::moduleHandler()->moduleExists('custom_module')) {
    $supported_entities['custom_entity_type'] = [
      'bundles' => [
        'default' => FALSE,
        'selected' => ['custom_bundle'],
      ],
      'languages' => [
        'default' => FALSE,
        'selected' => ['en'],
      ],
    ];
  }
  // Example: Disable Markdownify for taxonomy terms.
  if (isset($supported_entities['taxonomy_term'])) {
    // You can either unset it entirely or modify its configuration.
    $supported_entities['taxonomy_term']['bundles']['default'] = TRUE;
    $supported_entities['taxonomy_term']['languages']['default'] = TRUE;
  }
}

/**
 * Alter the render array of an entity before rendering to HTML.
 *
 * This hook allows other modules to modify the render array of an entity
 * before it is rendered to HTML during Markdown transformation.
 *
 * @param array &$build
 *   The render array for the entity.
 * @param array $context
 *   An associative array containing additional context:
 *   - entity: The entity being rendered (\Drupal\Core\Entity\EntityInterface).
 *   - view_mode: The view mode used to render the entity (string).
 *   - langcode: The language code for rendering (string).
 * @param \Drupal\Core\Render\BubbleableMetadata|null $metadata
 *   (optional) Cacheable metadata for the rendering process.
 */
function hook_markdownify_entity_build_alter(array &$build, array $context, ?BubbleableMetadata $metadata): void {
  // Example: Add a custom class to all entities of type 'node'.
  $entity = $context['entity'];
  // Example: Add metadata to cache tags for 'article' node bundles.
  if ($entity->getEntityTypeId() === 'node' && $entity->bundle() === 'article' && $metadata) {
    $metadata->addCacheTags(['custom_article_tag']);
  }
}

/**
 * Alter the rendered HTML of an entity before conversion to Markdown.
 *
 * This hook allows other modules to modify the HTML output of an entity
 * before it is passed to the Markdown converter.
 *
 * @param string &$html
 *   The rendered HTML of the entity.
 * @param array $context
 *   An associative array containing additional context:
 *   - entity: The entity being converted (\Drupal\Core\Entity\EntityInterface).
 *   - view_mode: The view mode used to render the entity (string).
 *   - langcode: The language code of the entity's content (string).
 * @param \Drupal\Core\Render\BubbleableMetadata|null $metadata
 *   (optional) Cacheable metadata for the rendering process.
 */
function hook_markdownify_entity_html_alter(string &$html, array $context, ?BubbleableMetadata $metadata): void {
  // Example: Add a custom wrapper around the HTML.
  $html = '<div class="custom-wrapper">' . $html . '</div>';
  // Example: Modify the HTML for a specific entity type.
  $entity = $context['entity'];
  if ($entity->getEntityTypeId() === 'node' && $entity->bundle() === 'article') {
    $html .= '<!-- Custom footer for article nodes -->';
  }
}

/**
 * Alter the Markdown output of an entity after conversion from HTML.
 *
 * This hook allows other modules to modify the Markdown output of an entity
 * after it has been converted from HTML.
 *
 * @param string &$markdown
 *   The Markdown output of the entity.
 * @param array $context
 *   An associative array containing additional context:
 *   - html: The original rendered HTML of the entity.
 */
function hook_markdownify_entity_markdown_alter(string &$markdown, array $context): void {
  // Example: Prepend a custom header to the Markdown output.
  if (strpos($context['html'], '<h1>') !== FALSE) {
    $markdown = "# Custom Header\n\n" . $markdown;
  }
}
