<?php

namespace Drupal\markdownify\Form;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a form for configuring supported entities, bundles, and languages.
 *
 * This form is used to specify which entity types and their associated bundles
 * and languages are supported for Markdown conversion.
 */
class SupportedEntitiesConfigForm implements PluginFormInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs a SupportedEntitiesConfigForm instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, LanguageManagerInterface $language_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Init the config form.
    $config_form = [];
    // Retrieve the config being edited.
    $supported_entities = $form_state->getValue('supported_entities', []);
    $previously_configured_entity_types = array_keys($supported_entities);
    $submitted_entity_types = $form_state->getUserInput()['entity_types'] ?? [];
    $submitted_entity_types = array_keys(array_filter($submitted_entity_types, fn($v) => !is_null($v)));
    $configured_entity_types = $submitted_entity_types ?: $previously_configured_entity_types;
    // Entity Configuration: Entity Types, Bundles, and Languages.
    $config_form['entities_configs'] = [
      '#type' => 'details',
      '#title' => $this->t('Entity Types and Bundle Configuration'),
      '#description' => $this->t('Select the entity types, bundles, and languages supports for Markdown conversion.'),
      '#open' => TRUE,
    ];
    // Retrieve content entity type options.
    $entity_types_options = $this->getContentEntityTypesOptions();
    $config_form['entities_configs']['entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Entity Types'),
      '#description' => $this->t('Enable conversion for selected entity types.'),
      '#options' => $entity_types_options['options'],
      '#default_value' => $configured_entity_types,
      '#required' => TRUE,
      '#ajax' => [
        'trigger_as' => ['name' => 'entities_configs_configure'],
        'callback' => [static::class, 'buildAjaxSupportedEntitiesConfigForm'],
        'wrapper' => 'supported-entities-config-form',
        'method' => 'replaceWith',
        'effect' => 'fade',
      ],
    ];
    // Add descriptions for each entity type.
    $config_form['entities_configs']['entity_types'] += $entity_types_options['descriptions'];
    // Supported entities configure (updated dynamically).
    $config_form['entities_configs']['supported_entities'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'supported-entities-config-form',
      ],
      '#tree' => TRUE,
    ];
    // Hidden submit used to rebuild AJAX.
    $config_form['entities_configs_configure_button'] = [
      '#type' => 'submit',
      '#name' => 'entities_configs_configure',
      '#value' => $this->t('Configure'),
      '#limit_validation_errors' => [['entities_configs']],
      '#submit' => [[static::class, 'submitAjaxSupportedEntitiesConfigForm']],
      '#ajax' => [
        'callback' => [static::class, 'buildAjaxSupportedEntitiesConfigForm'],
        'wrapper' => 'supported-entities-config-form',
      ],
      '#attributes' => ['class' => ['js-hide']],
    ];
    // Get the list of selected entity types.
    $selected_entity_types = array_filter($form_state->getValue('entity_types', $configured_entity_types));
    foreach ($selected_entity_types as $entity_type_id) {
      // Get the entity type definition.
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      // Determines whether the entity type supports bundles.
      $has_bundles = $this->hasBundles($entity_type);
      // Retrieves all bundles of this entity type.
      $bundle_options = $this->getEntityBundleOptions($entity_type);
      // Determines whether the entity type supports translations.
      $is_translatable = $entity_type->isTranslatable();
      // Determine if the configuration form applies for this entity type.
      $apply = $has_bundles || $is_translatable;
      // Early exit if this entity type is not configurable.
      if (!$apply) {
        continue;
      }
      // Wrap the entity type configuration inside a wrapper.
      $config_form['entities_configs']['supported_entities'][$entity_type_id] = [
        '#type' => 'details',
        '#title' => $this->t('Configure the "<i>@entity_type</i>" conversion support', ['@entity_type' => $entity_type->getLabel()]),
        '#open' => FALSE,
      ];
      // Configure bundles if applicable.
      if ($has_bundles && $bundle_options) {
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['bundles'] = [
          '#type' => 'details',
          '#title' => $this->t('Bundles'),
          '#open' => TRUE,
        ];
        $default_value = $supported_entities[$entity_type_id]['bundles']['default'] ?? 1;
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['bundles']['default'] = [
          '#type' => 'radios',
          '#title' => $this->t('Which bundles should be indexed?'),
          '#options' => [
            0 => $this->t('Only those selected'),
            1 => $this->t('All except those selected'),
          ],
          '#default_value' => (int) $default_value,
        ];
        $default_value = $supported_entities[$entity_type_id]['bundles']['selected'] ?? [];
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['bundles']['selected'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Bundles'),
          '#options' => $bundle_options,
          '#default_value' => $default_value,
          '#size' => min(4, count($bundle_options)),
          '#multiple' => TRUE,
        ];
      }
      // Configure language options if the entity type supports translation.
      if ($entity_type->isTranslatable()) {
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['languages'] = [
          '#type' => 'details',
          '#title' => $this->t('Languages'),
          '#open' => TRUE,
        ];
        $default_value = $supported_entities[$entity_type_id]['languages']['default'] ?? 1;
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['languages']['default'] = [
          '#type' => 'radios',
          '#title' => $this->t('Which languages should be indexed?'),
          '#options' => [
            0 => $this->t('Only those selected'),
            1 => $this->t('All except those selected'),
          ],
          '#default_value' => (int) $default_value,
        ];
        $default_value = $supported_entities[$entity_type_id]['languages']['selected'] ?? [];
        $config_form['entities_configs']['supported_entities'][$entity_type_id]['languages']['selected'] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Languages'),
          '#options' => $this->getTranslationOptions(),
          '#default_value' => $default_value,
          '#multiple' => TRUE,
        ];
      }
    }
    return $config_form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // Ensure 'supported_entities' exists in the form state before processing.
    if (!$form_state->hasValue('supported_entities')) {
      // No supported entities to process, exit early.
      return;
    }
    // Retrieve the configurations for supported entities from the form state.
    $supported_entities = $form_state->getValue('supported_entities', []);
    foreach ($supported_entities as $entity_type_id => $supported_entity) {
      foreach (['bundles', 'languages'] as $settings) {
        // Convert default to boolean value.
        if (isset($supported_entity[$settings]['default'])) {
          // Define the key used to update the value.
          $key = ['supported_entities', $entity_type_id, $settings, 'default'];
          // Convert the default value to a boolean.
          $value = (bool) $supported_entity[$settings]['default'];
          // Update the form state with the processed default value.
          $form_state->setValue($key, $value);
        }
        // Check if the key exists and has a 'selected' sub-key.
        if (isset($supported_entity[$settings]['selected']) && is_array($supported_entity[$settings]['selected'])) {
          // Define the key used to update the value.
          $key = ['supported_entities', $entity_type_id, $settings, 'selected'];
          // Filter out empty/unchecked values and retain the selected keys.
          $value = array_keys(array_filter($supported_entity[$settings]['selected']));
          // Update the form state with the filtered values.
          $form_state->setValue($key, $value);
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {}

  /**
   * Form submission handler for buildConfigurationForm().
   *
   * Takes care of changes in the selected supported entity-types.
   *
   * @param array $form
   *   The current form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public static function submitAjaxSupportedEntitiesConfigForm(array $form, FormStateInterface $form_state): void {
    $form_state->setRebuild(TRUE);
  }

  /**
   * Handles changes to the selected supported entity-types.
   *
   * @param array $form
   *   The current form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   *
   * @return array
   *   The part of the form to return as AJAX.
   */
  public static function buildAjaxSupportedEntitiesConfigForm(array $form, FormStateInterface $form_state): array {
    return $form['entities_configs']['supported_entities'];
  }

  /**
   * Retrieves the available languages of this entity type as an options list.
   *
   * @return array
   *   An associative array of language labels, keyed by the language name.
   */
  protected function getTranslationOptions(): array {
    $options = [];
    foreach ($this->languageManager->getLanguages() as $language) {
      $options[$language->getId()] = $language->getName();
    }
    return $options;
  }

  /**
   * Returns a list of available content entity types with descriptions.
   *
   * Only content entity types are returned, since configuration entities do not
   * utilize the render pipeline in a compatible way.
   *
   * @return array
   *   An associative array containing:
   *   - options: An array of entity type IDs and their labels.
   *   - descriptions: Descriptive strings for each entity type.
   */
  protected function getContentEntityTypesOptions(): array {
    $options = [];
    $descriptions = [];
    foreach ($this->entityTypeManager->getDefinitions() as $entity_type_id => $entity_type) {
      // We only support content entity types at the moment, since config
      // entities don't use the Drupal Render API and Drupal render pipeline.
      if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
        continue;
      }
      // Filter content entity types that does not support/implement a canonical
      // link template.
      if (!$entity_type->hasLinkTemplate('canonical')) {
        continue;
      }
      $options[$entity_type_id] = $entity_type->getLabel();
      $descriptions[$entity_type_id]['#description'] = $this->t('Allows conversion support for %entity_type entities.', ['%entity_type' => $entity_type->getLabel()]);
    }
    // Sort options alphabetically (case-insensitive).
    asort($options, SORT_NATURAL | SORT_FLAG_CASE);
    return [
      'options' => $options,
      'descriptions' => $descriptions,
    ];
  }

  /**
   * Retrieves the available bundles of this entity type as an options list.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return array
   *   An associative array of bundle labels, keyed by the bundle name.
   */
  protected function getEntityBundleOptions(EntityTypeInterface $entity_type): array {
    $options = [];
    if (($bundles = $this->getEntityBundles($entity_type))) {
      foreach ($bundles as $bundle => $bundle_info) {
        $options[$bundle] = $bundle_info['label'];
      }
    }
    return $options;
  }

  /**
   * Retrieves all bundles of the given entity type.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return array
   *   An associative array of bundle infos, keyed by the bundle names.
   */
  protected function getEntityBundles(EntityTypeInterface $entity_type): array {
    return $this->hasBundles($entity_type) ? $this->entityTypeBundleInfo->getBundleInfo($entity_type->id()) : [];
  }

  /**
   * Determines whether the entity type supports bundles.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return bool
   *   TRUE if the entity type supports bundles, FALSE otherwise.
   */
  protected function hasBundles(EntityTypeInterface $entity_type): bool {
    return $entity_type->hasKey('bundle');
  }

}
