<?php

namespace Drupal\marquee_scroll\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'marquee_scroll_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "marquee_scroll_formatter",
 *   label = @Translation("Marquee Scroll"),
 *   field_types = {
 *     "string",
 *     "string_long",
 *     "text",
 *     "text_long"
 *   }
 * )
 */
class MarqueeScrollFieldFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'speed' => 40,
      'gap' => 60,
      'direction' => 'left',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $element['speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Speed (px/s)'),
      '#default_value' => (int) $this->getSetting('speed'),
      '#min' => 5,
      '#max' => 500,
      '#description' => $this->t('Scrolling speed in pixels per second.'),
    ];
    $element['gap'] = [
      '#type' => 'number',
      '#title' => $this->t('Gap (px)'),
      '#default_value' => (int) $this->getSetting('gap'),
      '#min' => 0,
      '#max' => 1000,
      '#description' => $this->t('Spacing after each item before repeat.'),
    ];
    $element['direction'] = [
      '#type' => 'select',
      '#title' => $this->t('Direction'),
      '#options' => [
        'left' => $this->t('Left'),
        'right' => $this->t('Right'),
        'up' => $this->t('Up'),
        'down' => $this->t('Down'),
      ],
      '#default_value' => $this->getSetting('direction') ?: 'left',
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Speed: @s px/s', ['@s' => (int) $this->getSetting('speed')]);
    $summary[] = $this->t('Gap: @g px', ['@g' => (int) $this->getSetting('gap')]);
    $summary[] = $this->t('Direction: @d', ['@d' => $this->getSetting('direction')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $speed = (int) $this->getSetting('speed');
    $gap = (int) $this->getSetting('gap');
    $direction = $this->getSetting('direction') ?: 'left';

    foreach ($items as $delta => $item) {
      $text = Html::escape($item->value);
      $elements[$delta] = [
        '#type' => 'inline_template',
        '#template' => '<div class="marquee-scroll-wrap" data-ms-speed="{{ speed }}" data-ms-gap="{{ gap }}" data-ms-direction="{{ direction }}"><div class="marquee-scroll-track"><span class="marquee-scroll-item">{{ text }}</span><span class="marquee-scroll-item">{{ text }}</span><span class="marquee-scroll-item">{{ text }}</span></div></div>',
        '#context' => [
          'text' => $text,
          'speed' => $speed,
          'gap' => $gap,
          'direction' => $direction,
        ],
        '#attached' => [
          'library' => ['marquee_scroll/scroll'],
          'drupalSettings' => [
            'marquee_scroll' => [
              'speed' => $speed,
              'gap' => $gap,
              'direction' => $direction,
            ],
          ],
        ],
      ];
    }

    return $elements;
  }

}
