<?php

declare(strict_types=1);

namespace Drupal\masquerade_field\Hook;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\UserInterface;

/**
 * Hook implementation for the Masquerade Field module.
 */
class MasqueradeFieldHooks {

  use StringTranslationTrait;

  /**
   * Constructs a new MasqueradeFieldHooks instance.
   */
  public function __construct(protected EntityTypeManagerInterface $entityTypeManager) {}

  /**
   * Adds a masquerade field to the user entity.
   */
  #[Hook('entity_base_field_info')]
  public function entityBaseFieldInfo(EntityTypeInterface $entity_type): array {
    $fields = [];
    if ($entity_type->id() === 'user') {
      $fields['masquerade_as'] = BaseFieldDefinition::create('masquerade_field')
        ->setLabel($this->t('Masquerade as'))
        ->setDescription($this->t('Select the accounts this user can masquerade as.'))
        ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
        ->setDisplayConfigurable('form', TRUE)
        ->setDisplayOptions('form', [
          'type' => 'entity_reference_autocomplete',
        ])
        ->setDisplayConfigurable('view', TRUE)
        ->setDisplayOptions('view', [
          'type' => 'masquerade_field_default',
        ])
        ->setConstraints(['ExcludeOriginUser' => []])
        ->setTranslatable(FALSE);
    }

    return $fields;
  }

  /**
   * Sets the default widget for the masquerade field.
   */
  #[Hook('field_widget_info_alter')]
  public function fieldWidgetInfoAlter(array &$info): void {
    $info['entity_reference_autocomplete']['field_types'][] = 'masquerade_field';
  }

  /**
   * Implements access control for the masquerade field.
   */
  #[Hook('entity_field_access')]
  public function entityFieldAccess(string $operation, FieldDefinitionInterface $field_definition, AccountInterface $account, ?FieldItemListInterface $items = NULL): AccessResultInterface {
    if ($field_definition->getTargetEntityTypeId() !== 'user' || $field_definition->getName() !== 'masquerade_as') {
      // Let other modules decide.
      return AccessResult::neutral();
    }

    if ($operation === 'view') {
      $permitted = $account->hasPermission('view any masquerade field') || ($account->id() === $items->getEntity()->id() && $account->hasPermission('view own masquerade field'));
      $access_result = AccessResult::allowedIf($permitted)->addCacheContexts(['user.permissions']);
    }
    elseif ($operation === 'edit') {
      $access_result = AccessResult::allowedIfHasPermission($account, 'edit masquerade field');
    }
    else {
      throw new \InvalidArgumentException("Wrong operation '$operation'.");
    }

    // By default, this field cannot be accessed.
    return $access_result->isAllowed() ? $access_result : AccessResult::forbidden()->addCacheContexts(['user.permissions']);
  }

  /**
   * Extends the masquerade access check to include the masquerade field.
   */
  #[Hook('masquerade_access')]
  public function masqueradeAccess(AccountProxyInterface $user, UserInterface $target_account): ?bool {
    // Load the user account entity.
    $account = $this->entityTypeManager->getStorage('user')->load($user->id());
    $uids = [];
    foreach ($account->get('masquerade_as') as $field_item) {
      $uids[$field_item->target_id] = TRUE;
    }

    // The current user is explicitly configured to masquerade as target
    // account.
    if (isset($uids[$target_account->id()])) {
      return TRUE;
    }

    // Let the other modules decide.
    return NULL;
  }

}
