<?php

namespace Drupal\matrix_api\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\matrix_api\MatrixClientInterface;

/**
 * Provides a a Send a message to a Matrix room action.
 *
 * @Action(
 *   id = "matrix_api_send_message",
 *   label = @Translation("Send Matrix Message"),
 *   description = @Translation("Sends a message to a specified Matrix room."),
 *   type = "matrix"
 * )
 *
 */
class SendMessage extends ConfigurableActionBase {

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    /** @var MatrixClientInterface $matrixClient */
    $matrixClient = \Drupal::service('matrix_api.matrixclient');
    if (\Drupal::moduleHandler()->moduleExists('eca')) {
      /** @var \Drupal\eca\Token\TokenServices $tokenServices */
      $tokenServices = \Drupal::service('eca.token_services');
      $room = $tokenServices->replaceClear($this->configuration['room']);
      $html_message = $tokenServices->replaceClear($this->configuration['html_message']);
      $message = $tokenServices->replaceClear($this->configuration['message']);
    }
    else {
      $room = $this->configuration['room'];
      $html_message = $this->configuration['html_message'];
      $message = $this->configuration['message'];
    }
    $roomId = $matrixClient->join($room);
    $msgtype = $this->configuration['msgtype'];

    if (empty($html_message) && empty($msgtype)) {
      $data = $message;
    } else {
      if (empty($message)) {
        $message = strip_tags($html_message);
      }
      if (empty($msgtype)) {
        $msgtype = 'm.notice';
      }
      $data = [
        'format' => 'org.matrix.custom.html',
        'formatted_body' => $html_message,
        'body' => $message,
        'msgtype' => $msgtype,
      ];
    }
    if (empty($data)) {
      return;
    }

    $matrixClient->sendMessage($roomId, $data);
  }

  /**
   * {@inheritDoc}
   */
  public function defaultConfiguration(): array
  {
    return [
      'room' => '',
      'html_message' => '',
      'message' => '',
      'msgtype' => 'm.notice',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['room'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Room'),
      '#default_value' => $this->configuration['room'],
      '#description' => $this->t('The Room alias or id to send the message.'),
      '#weight' => -30,
    ];

    $form['html_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('HTML Message'),
      '#description' => $this->t('The message to send - HTML. Omitted if blank.'),
      '#default_value' => $this->configuration['html_message'],
      '#weight' => -10,
    ];
    $form['message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Text Message'),
      '#description' => $this->t('The text message to send. If empty, uses the HTML message stripped of tags.'),
      '#default_value' => $this->configuration['message'],
      '#weight' => -10,
    ];

    $form['msgtype'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Msg Type'),
      '#default_value' => $this->configuration['msgtype'],
      '#description' => $this->t('The type of message to send -- "m.notice" is recommended for bot messasges, "m.text"
         for regular messages.'),
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['room'] = $form_state->getValue('room');
    $this->configuration['message'] = $form_state->getValue('message');
    $this->configuration['html_message'] = $form_state->getValue('html_message');
    $this->configuration['msgtype'] = $form_state->getValue('msgtype');
  }

  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = AccessResult::allowed();
    return $return_as_object ? $result : $result->isAllowed();
  }
}
