<?php

declare(strict_types=1);

namespace Drupal\mautic_api\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the Mautic API Connection entity type.
 *
 * @ConfigEntityType(
 *   id = "mautic_api_connection",
 *   label = @Translation("Mautic API Connection"),
 *   label_collection = @Translation("Mautic API Connections"),
 *   label_singular = @Translation("Mautic API Connection"),
 *   label_plural = @Translation("Mautic API Connections"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Mautic API Connection",
 *     plural = "@count Mautic API Connections",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\mautic_api\MauticApiConnectionListBuilder",
 *     "form" = {
 *       "add" = "Drupal\mautic_api\Form\MauticApiConnectionForm",
 *       "edit" = "Drupal\mautic_api\Form\MauticApiConnectionForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "mautic_api_connection",
 *   admin_permission = "administer mautic_api_connection",
 *   links = {
 *     "collection" = "/admin/structure/mautic-api-connection",
 *     "add-form" = "/admin/structure/mautic-api-connection/add",
 *     "edit-form" = "/admin/structure/mautic-api-connection/{mautic_api_connection}",
 *     "delete-form" = "/admin/structure/mautic-api-connection/{mautic_api_connection}/delete",
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "http_protocol",
 *     "base_url",
 *     "port",
 *     "path",
 *     "auth_method",
 *     "public_key",
 *     "secret_key",
 *     "basic_auth_username",
 *     "basic_auth_password",
 *   },
 * )
 */
final class MauticApiConnection extends ConfigEntityBase implements MauticApiConnectionInterface {

  /**
   * The connection ID.
   */
  protected string $id;

  /**
   * The connection label.
   */
  protected string $label;

  /**
   * The connection description.
   */
  protected string $description;

  /**
   * The HTTP protocol (http/https).
   */
  protected string $http_protocol;

  /**
   * The Mautic base URL.
   */
  protected string $base_url;

  /**
   * The Mautic port.
   */
  protected string $port;

  /**
   * The Mautic path.
   */
  protected string $path;

  /**
   * The authentication method (oauth/basic).
   */
  protected string $auth_method;

  /**
   * The OAuth public key.
   */
  protected string $public_key;

  /**
   * The OAuth secret key.
   */
  protected string $secret_key;

  /**
   * The Basic Auth username.
   */
  protected string $basic_auth_username;

  /**
   * The Basic Auth password.
   */
  protected string $basic_auth_password;

  /**
   * {@inheritdoc}
   */
  public function getHttpProtocol(): string {
    return $this->http_protocol;
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseUrl(): string {
    return $this->base_url;
  }

  /**
   * {@inheritdoc}
   */
  public function getPort(): string {
    return $this->port;
  }

  /**
   * {@inheritdoc}
   */
  public function getPath(): string {
    return $this->path;
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthMethod(): string {
    return $this->auth_method;
  }

  /**
   * {@inheritdoc}
   */
  public function getPublicKey(): string {
    return $this->public_key;
  }

  /**
   * {@inheritdoc}
   */
  public function getSecretKey(): string {
    return $this->secret_key;
  }

  /**
   * {@inheritdoc}
   */
  public function getBasicAuthUsername(): string {
    return $this->basic_auth_username;
  }

  /**
   * {@inheritdoc}
   */
  public function getBasicAuthPassword(): string {
    return $this->basic_auth_password;
  }

  /**
   * {@inheritdoc}
   */
  public function getCompleteUrl(): string {
    $base_url = $this->getHttpProtocol() . '://' . $this->getBaseUrl();

    if ($this->getPort()) {
      $base_url .= ':' . $this->getPort();
    }

    if ($this->getPath()) {
      $base_url .= '/' . trim($this->getPath(), '/');
    }

    return $base_url;
  }

}
