<?php

declare(strict_types=1);

namespace Drupal\mautic_api;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\mautic_api\Entity\MauticApiConnection;
use Drupal\mautic_api\Entity\MauticApiConnectionInterface;
use Mautic\Api\Api;
use Mautic\Auth\ApiAuth;
use Mautic\Auth\TwoLeggedOAuth2;
use Mautic\MauticApi;
use Symfony\Component\HttpFoundation\RequestStack;
use Psr\Log\LoggerInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for handling Mautic API connections and requests.
 */
class MauticApiConnector {

  /**
   * The Mautic API factory.
   */
  protected MauticApi $mauticApi;

  /**
   * Constructs a new MauticApiConnector object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RequestStack $requestStack,
    protected LoggerInterface $logger,
    protected StateInterface $state,
  ) {
    $this->mauticApi = new MauticApi();
  }

  /**
   * Get a Mautic API context object.
   *
   * @param \Drupal\mautic_api\Entity\MauticApiConnectionInterface $connection
   *   The Mautic API connection configuration entity.
   * @param string $apiContext
   *   The Mautic API context.
   *
   * @return \Mautic\Api\Api
   *   The API object with the specified connection.
   */
  public function getApi(MauticApiConnectionInterface $connection, string $apiContext = 'contacts'): Api {
    $initAuth = new ApiAuth();
    $settings = $this->prepSettings($connection);

    if ($connection->getAuthMethod() === 'oauth') {
      /** @var \Mautic\Auth\TwoLeggedOAuth2 $auth */
      $auth = $initAuth->newAuth($settings, 'TwoLeggedOAuth2');

      $token_status = $this->getOauthAccessToken($auth, $connection->id());
      if (!$token_status['success']) {
        $this->logger->error($token_status['message']);
      }

    }
    else {
      $auth = $initAuth->newAuth($settings, 'BasicAuth');
    }

    try {
      return $this->mauticApi->newApi($apiContext, $auth, $connection->getCompleteUrl());
    }
    catch (\Exception $e) {
      $this->logger->error('API client initialization failed: @error', ['@error' => $e->getMessage()]);
      throw new \RuntimeException("Failed to initialize API client: {$e->getMessage()}", 0, $e);
    }

  }

  /**
   * Get the status of the Mautic API connection.
   *
   * @param \Drupal\mautic_api\Entity\MauticApiConnection $connection
   *   The Mautic API connection entity.
   *
   * @return array
   *   An array containing:
   *   - status_code: The HTTP status code if the request was successful,
   *     NULL otherwise
   *   - message: A descriptive message about the status or error
   */
  public function getStatus(MauticApiConnection $connection): array {
    try {
      $api = $this->getApi($connection);
      $response = $api->getList('', 0, 1, '', 'ASC', FALSE, TRUE);
      if ($response) {
        $responseInfo = $api->getResponseInfo();
        $statusCode = $responseInfo['http_code'] ?? NULL;
        if ($statusCode === 200) {
          $message = "Successfully connected to the Mautic API instance. <b>STATUS: 200 OK</b.";
        }
        else {
          $message = "Mautic API connection <b>FAILED</b>. Please check the logs for more information.";
        }
      }
      else {
        $message = "<b>NO RESPONSE RECEIVED</b> from the Mautic API. Please check your connection settings.";
      }
    }
    catch (\Exception $e) {
      $message = $e->getMessage();
      $this->logger->error('Mautic API status check failed: @error', ['@error' => $message]);
    }

    return [
      'status_code' => $statusCode,
      'message' => $message,
    ];
  }

  /**
   * Prepare the settings for the Mautic auth objects.
   *
   * @param \Drupal\mautic_api\Entity\MauticApiConnection $connection
   *   The Mautic API connection entity.
   *
   * @return array
   *   The prepared settings array.
   */
  protected function prepSettings(MauticApiConnection $connection): array {
    $settings = [
      'baseUrl' => $connection->getCompleteUrl(),
    ];

    if ($connection->getAuthMethod() === 'oauth') {
      $settings += [
        'version'          => 'OAuth2',
        'clientKey'        => $connection->getPublicKey(),
        'clientSecret'     => $connection->getSecretKey(),
        'callback' => $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost() . $this->requestStack->getCurrentRequest()->getRequestUri(),
      ];
    }
    else {
      $settings += [
        'userName'   => $connection->getBasicAuthUsername(),
        'password'   => $connection->getBasicAuthPassword(),
      ];
    }
    return $settings;
  }

  /**
   * Get and manage OAuth access token for a specific connection.
   *
   * @param \Mautic\Auth\TwoLeggedOAuth2 $auth
   *   The TwoLeggedOAuth2 auth object.
   * @param string $connection_id
   *   The Mautic API connection configuration entity ID.
   *
   * @return array
   *   The status of the access token operation.
   */
  public function getOauthAccessToken(TwoLeggedOAuth2 $auth, string $connection_id): array {
    $state_key = 'mautic_api.oauth_access_tokens';
    $tokens = $this->state->get($state_key, []);

    if (!empty($tokens[$connection_id])) {
      $auth->setAccessTokenDetails($tokens[$connection_id]);
    }

    if (!$auth->validateAccessToken()) {
      $auth->setAccessTokenDetails(['access_token' => '']);
      try {
        $auth->requestAccessToken();
        $accessTokenData = $auth->getAccessTokenData();

        $tokens[$connection_id] = $accessTokenData;
        $this->state->set($state_key, $tokens);

        return [
          'success' => TRUE,
          'message' => 'Successfully obtained access token.',
        ];
      }
      catch (\Exception $e) {
        $error_message = $e->getMessage();
        if (strpos($error_message, 'invalid_client') !== FALSE) {
          return [
            'success' => FALSE,
            'message' => 'Failed to obtain the OAuth Access Token. Please verify your client ID and client secret.',
          ];
        }
        return [
          'success' => FALSE,
          'message' => 'Failed to obtain Mautic API access token: ' . $error_message,
        ];
      }
    }

    return [
      'success' => TRUE,
      'message' => 'Using existing valid access token.',
    ];
  }

  /**
   * Remove OAuth access token for a specific connection.
   *
   * @param string $connection_id
   *   The Mautic API connection configuration entity ID.
   */
  public function removeOauthAccessToken(string $connection_id): void {
    $state_key = 'mautic_api.oauth_access_tokens';
    $tokens = $this->state->get($state_key, []);

    if (isset($tokens[$connection_id])) {
      unset($tokens[$connection_id]);
      $this->state->set($state_key, $tokens);
      $this->logger->info('Cleared OAuth access token for Mautic API connection: @id', ['@id' => $connection_id]);
    }
  }

}
