<?php

declare(strict_types=1);

namespace Drupal\mautic_api\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\mautic_api\Entity\MauticApiConnection;
use Drupal\mautic_api\MauticApiConnector;
use Mautic\Auth\ApiAuth;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Mautic API Connection entity add/edit forms.
 */
class MauticApiConnectionForm extends EntityForm {

  /**
   * Constructs a new MauticApiConnectionForm object.
   */
  public function __construct(
    protected MauticApiConnector $mauticApiConnector,
    protected $requestStack,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('mautic_api.connector'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\mautic_api\Entity\MauticApiConnection $entity */
    $entity = $this->entity;

    if (!$entity->isNew()) {
      $statusInfo = $this->mauticApiConnector->getStatus($entity);
      $form['connection_status'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Status'),
        '#title_display' => 'before',
        '#open' => TRUE,
        '#description' => $statusInfo['message'],
      ];
    }

    $form['instructions'] = [
      '#type' => 'details',
      '#title' => $this->t("Instructions"),
      '#open' => FALSE,
      '#description' => [
        '#type' => 'inline_template',
        '#template' => "<ol>
          <li><strong>{{ step1_title }}</strong>
            <ul>
              <li>{{ step1_item1 }}</li>
              <li>{{ step1_item2 }}</li>
              <li>{{ step1_item3 }}</li>
              <li>{{ step1_item4 }}</li>
            </ul>
          </li>
          <li><strong>{{ step2_title }}</strong>
            <ul>
              <li>{{ step2_item1 }}</li>
              <li>{{ step2_item2 }}</li>
              <li>{{ step2_item3 }}</li>
              <li>{{ step2_item4 }}</li>
            </ul>
          </li>
          <li><strong>{{ step3_title }}</strong>
            <ul>
              <li>{{ step3_item1 }}</li>
              <li>{{ step3_item2 }}</li>
              <li>{{ step3_item3 }}</li>
              <li>{{ step3_item4 }}</li>
            </ul>
          </li>
          <li><strong>{{ step4_title }}</strong>
            <ul>
              <li>{{ step4_item1 }}</li>
              <li>{{ step4_item2 }}</li>
              <li>{{ step4_item3 }}</li>
            </ul>
          </li>
        </ol>",
        '#context' => [
          'step1_title' => $this->t("Enable the API in Mautic:"),
          'step1_item1' => $this->t("Navigate to your respective Mautic site and log in as an administrator."),
          'step1_item2' => $this->t("Go to Settings → Configuration → API Settings."),
          'step1_item3' => $this->t("Find the API Settings section and enable the API."),
          'step1_item4' => $this->t("Save the configuration changes."),
          'step2_title' => $this->t("Create New API Credentials:"),
          'step2_item1' => $this->t("In your Mautic site, go to Settings → API Credentials."),
          'step2_item2' => $this->t("Click 'New' to create new API credentials."),
          'step2_item3' => $this->t("Give your new credentials a descriptive name (e.g., 'Drupal Integration')."),
          'step2_item4' => $this->t("Set the Redirect URI to your Drupal website's main domain (e.g., https://www.your-drupal-site.com)."),
          'step3_title' => $this->t("Connect to Drupal:"),
          'step3_item1' => $this->t("After saving, Mautic will generate a Client ID (Public Key) and a Client Secret."),
          'step3_item2' => $this->t("Copy both of these values and paste them into the corresponding fields below."),
          'step3_item3' => $this->t("Ensure your Mautic Base URL is correctly set to your Mautic site's domain."),
          'step3_item4' => $this->t("If your Mautic site uses a custom path, include it in the Path field."),
          'step4_title' => $this->t("Test the Connection:"),
          'step4_item1' => $this->t("After saving the Drupal configuration, the system will attempt to authenticate with Mautic."),
          'step4_item2' => $this->t("Check the status section above to confirm the connection is working properly."),
          'step4_item3' => $this->t("If authentication fails, verify your credentials and Mautic site configuration."),
        ],
      ],
    ];

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('Label for the Mautic API Connection.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [MauticApiConnection::class, 'load'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#description' => $this->t('Description of the Mautic API Connection.'),
    ];

    $form['http_protocol'] = [
      '#type' => 'select',
      '#title' => $this->t('HTTP Protocol'),
      '#options' => [
        'http' => $this->t('HTTP'),
        'https' => $this->t('HTTPS'),
      ],
      '#default_value' => $entity->get('http_protocol') ?: 'https',
      '#required' => TRUE,
      '#description' => $this->t('The HTTP protocol to use for sending queries.'),
    ];

    $form['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mautic Base URL'),
      '#default_value' => $entity->get('base_url'),
      '#required' => TRUE,
      '#description' => $this->t('The base url name or IP of your Mautic server, e.g. localhost or example.com.'),
    ];

    $form['port'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mautic Port'),
      '#default_value' => $entity->get('port'),
      '#description' => $this->t('An example port can be 3306.'),
    ];

    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mautic Path'),
      '#default_value' => $entity->get('path'),
      '#description' => $this->t('The path that identifies the Mautic instance to use on the server.'),
    ];

    $form['auth_method'] = [
      '#type' => 'radios',
      '#title' => $this->t('Authentication Method'),
      '#options' => [
        'oauth' => $this->t('OAuth'),
        'basic' => $this->t('Basic Auth'),
      ],
      '#default_value' => $entity->get('auth_method') ?: 'oauth',
      '#required' => TRUE,
      '#description' => $this->t('Choose a connector to use for this Mautic server.'),
    ];

    $form['oauth_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Mautic OAuth Integration Settings'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="auth_method"]' => ['value' => 'oauth'],
        ],
      ],
    ];

    $form['oauth_settings']['public_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public Key'),
      '#default_value' => $entity->get('public_key'),
      '#description' => $this->t('You can generate and view this in your Mautic instance at <code>/s/credentials</code>. This is a required field.'),
    ];

    $form['oauth_settings']['secret_key'] = [
      '#type' => 'password',
      '#title' => $this->t('Secret Key'),
      '#description' => $this->t('While editing an existing connection, if this field is left blank and the Public Key is filled out, the current Secret Key will not be changed.'),
    ];
    $form['oauth_settings']['secret_key']['#old-value'] = $this->entity->get('secret_key');

    $form['basic_auth_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('HTTP Basic Authentication'),
      '#open' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="auth_method"]' => ['value' => 'basic'],
        ],
      ],
    ];

    $form['basic_auth_settings']['basic_auth_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $entity->get('basic_auth_username'),
      '#description' => $this->t('The username to use for the HTTP Basic Authentication.'),
    ];

    $form['basic_auth_settings']['basic_auth_password'] = [
      '#type' => 'password',
      '#title' => $this->t('Password'),
      '#description' => $this->t('While editing an existing connection, if this field is left blank and the HTTP username is filled out, the current password will not be changed.'),
    ];
    $form['basic_auth_settings']['basic_auth_password']['#old-value'] = $this->entity->get('basic_auth_password');

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $entity = $this->entity;
    $auth_method = $form_state->getValue('auth_method');

    // Handle secret key preservation for OAuth.
    $old_secret_key = $form['oauth_settings']['secret_key']["#old-value"];
    if (empty($form_state->getValue('secret_key')) && $old_secret_key) {
      $entity->set('secret_key', $old_secret_key);
    }

    // Handle password preservation for Basic Auth.
    $old_basic_auth_password = $form['basic_auth_settings']['basic_auth_password']["#old-value"];
    if (empty($form_state->getValue('basic_auth_password')) && $old_basic_auth_password) {
      $entity->set('basic_auth_password', $old_basic_auth_password);
    }

    // Clear unused fields based on selected auth method.
    if ($auth_method === 'oauth') {
      $entity->set('basic_auth_username', '');
      $entity->set('basic_auth_password', '');
    }
    elseif ($auth_method === 'basic') {
      $entity->set('public_key', '');
      $entity->set('secret_key', '');
    }

    $status = $entity->save();

    if ($status) {
      $this->messenger()->addMessage($this->t('Saved the %label Mautic API Connection.', [
        '%label' => $entity->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('The %label Mautic API Connection was not saved.', [
        '%label' => $entity->label(),
      ]), 'error');
    }

    $form_state->setRedirect('entity.mautic_api_connection.collection');

    return parent::save(
      $form,
      $form_state
    );
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm(
      $form,
      $form_state
    );

    // Save the Access Token for OAuth method.
    if ($form_state->getValue('auth_method') === 'oauth') {
      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $entity */
      $entity = $this->entity;

      $new_secret_key = $form_state->getValue('secret_key');
      $old_secret_key = $form['oauth_settings']['secret_key']['#old-value'] ?? '';

      $secret_key = !empty($new_secret_key) ? $new_secret_key : $old_secret_key;

      // Ensure we have a secret key before proceeding.
      if (empty($secret_key)) {
        $this->messenger()->addMessage($this->t('Secret Key is required for OAuth authentication.'), 'error');
        return;
      }

      $settings = [
        'baseUrl' => $entity->getCompleteUrl(),
        'version' => 'OAuth2',
        'clientKey' => $form_state->getValue('public_key'),
        'clientSecret' => $secret_key,
      ];

      $initAuth = new ApiAuth();
      /** @var \Mautic\Auth\TwoLeggedOAuth2 $auth */
      $auth = $initAuth->newAuth($settings, 'TwoLeggedOAuth2');

      // Initialize OAuth token and get status.
      $token_status = $this->mauticApiConnector->getOauthAccessToken($auth, $entity->id());

      if (!$token_status['success']) {
        $this->messenger()->addWarning(($token_status['message']));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $auth_method = $form_state->getValue('auth_method');
    $is_new = $this->entity->isNew();

    if ($auth_method === 'oauth') {
      if (empty($form_state->getValue('public_key'))) {
        $form_state->setError($form['oauth_settings']['public_key'], $this->t('Public Key is required for OAuth authentication.'));
      }
      if ($is_new && empty($form_state->getValue('secret_key'))) {
        $form_state->setError($form['oauth_settings']['secret_key'], $this->t('Secret Key is required for OAuth authentication.'));
      }
    }
    elseif ($auth_method === 'basic') {
      if (empty($form_state->getValue('basic_auth_username'))) {
        $form_state->setError($form['basic_auth_settings']['basic_auth_username'], $this->t('Username is required for Basic Authentication.'));
      }
      if ($is_new && empty($form_state->getValue('basic_auth_password'))) {
        $form_state->setError($form['basic_auth_settings']['basic_auth_password'], $this->t('Password is required for Basic Authentication.'));
      }
    }
  }

}
