<?php

declare(strict_types=1);

namespace Drupal\mautic_api\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\mautic_api\Entity\Webhook;

/**
 * Form handler for the Webhook config entity add/edit forms.
 */
class WebhookForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);
    /** @var \Drupal\mautic_api\Entity\Webhook $entity */
    $entity = $this->entity;
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('Name for the Webhook.'),
      '#required' => TRUE,
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => [Webhook::class, 'load'],
      ],
      '#disabled' => !$entity->isNew(),
    ];
    $form['secret'] = [
      '#type' => 'password',
      '#title' => $this->t('Secret'),
      '#description' => $this->t('Secret from the Mautic created webhook.'),
    ];
    $form['secret']['#old-value'] = $entity->get('secret');
    $form['url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webhook Source URL'),
      '#default_value' => $entity->get('url'),
      '#description' => $this->t('The URL from which this webhook POSTs (the Mautic instance sending the request).'),
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $entity = $this->entity;
    $old_secret = $form['secret']['#old-value'];
    if (empty($form_state->getValue('secret')) && $old_secret) {
      $entity->set('secret', $old_secret);
    }
    $status = $entity->save();
    if ($status) {
      $this->messenger()->addMessage($this->t('Saved the %label Webhook.', [
        '%label' => $entity->label(),
      ]));
    }
    else {
      $this->messenger()->addMessage($this->t('The %label Webhook was not saved.', [
        '%label' => $entity->label(),
      ]), 'error');
    }
    $form_state->setRedirect('entity.mautic_api_webhook.collection');
    return parent::save($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    $is_new = $this->entity->isNew();
    $new_secret = $form_state->getValue('secret');
    $old_secret = $form['secret']['#old-value'] ?? '';
    if ($is_new && empty($new_secret)) {
      $form_state->setError($form['secret'], $this->t('Secret is required for new webhooks.'));
    }
    elseif (!$is_new && empty($new_secret) && empty($old_secret)) {
      $form_state->setError($form['secret'], $this->t('Secret is required.'));
    }
  }

}
