<?php

declare(strict_types=1);

namespace Drupal\mautic_api\Plugin\rest\resource;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\mautic_api\Event\MauticWebhookEvent;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Represents Mautic Webhook Event records as resources.
 */
#[RestResource(
  id: 'mautic_api_webhook_event',
  label: new TranslatableMarkup('Mautic Webhook Event'),
  uri_paths: [
    'create' => '/mautic-webhook-event',
  ],
)]
class MauticWebhookEventResource extends ResourceBase {

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->eventDispatcher = $container->get('event_dispatcher');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * Responds to POST requests and saves the new record.
   */
  public function post(Request $request): ResourceResponse {
    $origin_url = $request->headers->get('x-origin-base-url');

    // Find the Webhook config entity with a matching URL.
    $storage = $this->entityTypeManager->getStorage('mautic_api_webhook');
    $webhooks = $storage->loadByProperties(['url' => $origin_url]);

    // Making a assumption that
    // there is only one webhook for a given source URL.
    /** @var \Drupal\mautic_api\Entity\WebhookInterface|null $webhook */
    $webhook = $webhooks ? reset($webhooks) : NULL;
    if (!$webhook) {
      return new ResourceResponse([
        'error' => 'No webhook config found for this source URL',
        'source_url' => $origin_url,
      ], 401);
    }

    $secret = $webhook->getSecret();
    $rawData = $request->getContent();
    $receivedSignature = $request->headers->get('Webhook-Signature');
    // Compute the expected signature using HMAC-SHA256.
    $computedSignature = base64_encode(hash_hmac('sha256', $rawData, $secret, TRUE));
    if ($receivedSignature !== $computedSignature) {
      return new ResourceResponse(['error' => 'Invalid signature'], 401);
    }

    $content = json_decode($rawData, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new ResourceResponse(['error' => 'Invalid JSON data'], 400);
    }

    $webhookEvent = new MauticWebhookEvent($content);
    $this->eventDispatcher->dispatch($webhookEvent, 'mautic_api.webhook_received');

    return new ResourceResponse(['status' => 'success'], 200);
  }

}
