<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\mautic_api\MauticApiConnector;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Mautic ECA actions.
 */
abstract class MauticActionBase extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The Mautic API connector service.
   *
   * @var \Drupal\mautic_api\MauticApiConnector
   */
  protected MauticApiConnector $mauticApiConnector;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->mauticApiConnector = $container->get('mautic_api.connector');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mautic_connection' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $connection_options = [];
    try {
      $connections = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->loadMultiple();

      foreach ($connections as $id => $connection) {
        $connection_options[$id] = $connection->label();
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to load Mautic connections: @error', [
        '@error' => $e->getMessage(),
      ]);
    }

    $form['mautic_connection'] = [
      '#type' => 'select',
      '#title' => $this->t('Mautic connection'),
      '#description' => $this->t('Select the Mautic connection to use for this action.'),
      '#options' => $connection_options,
      '#default_value' => $this->configuration['mautic_connection'],
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select a connection -'),
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name of token'),
      '#default_value' => $this->configuration['token_name'],
      '#weight' => -30,
      '#description' => $this->t('Provide the name of a token where the value should be stored.'),
      '#eca_token_reference' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['mautic_connection'] = $form_state->getValue('mautic_connection');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
    parent::submitConfigurationForm($form, $form_state);
  }

}
