<?php

namespace Drupal\mautic_eca\Plugin\ECA\Event;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Attributes\Token;
use Drupal\eca\Event\Tag;
use Drupal\eca\Plugin\ECA\Event\EventBase;
use Drupal\eca\Plugin\DataType\DataTransferObject;
use Drupal\mautic_eca\Event\MauticWebhookReceived;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for the Mautic Webhook Events.
 *
 * @EcaEvent(
 *   id = "mautic_webhook_eca",
 *   deriver = "Drupal\mautic_eca\Plugin\ECA\Event\MauticWebhookEventDeriver",
 *   description = @Translation("All events for the Mautic webhook received."),
 *   eca_version_introduced = "1.0.0"
 * )
 */
class MauticWebhookEvent extends EventBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function definitions(): array {
    $definitions = [];
    $definitions['webhook_received'] = [
      'label' => 'Mautic: Webhook Received Event',
      'description' => 'When a Mautic webhook is received.',
      'event_name' => 'mautic_eca.webhook_received',
      'event_class' => MauticWebhookReceived::class,
      'tags' => Tag::RUNTIME | Tag::AFTER,
    ];
    return $definitions;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'webhook_event_type' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    // Load available webhook config entities.
    $webhook_options = [];
    try {
      $storage = $this->entityTypeManager->getStorage('mautic_api_webhook');
      $webhooks = $storage->loadMultiple();

      foreach ($webhooks as $webhook) {
        /** @var \Drupal\mautic_api\Entity\WebhookInterface $webhook */
        $webhook_options[$webhook->id()] = $webhook->label() . ' (' . $webhook->getUrl() . ')';
      }
    }
    catch (\Exception $e) {
      $webhook_options = [];
    }

    $form['webhook_event_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Webhook Configuration'),
      '#description' => $this->t('Select which webhook configuration to listen for.'),
      '#options' => $webhook_options,
      '#empty_option' => $this->t('- Select a webhook -'),
      '#default_value' => $this->configuration['webhook_event_type'] ?? '',
      '#required' => TRUE,
    ];

    if (empty($webhook_options)) {
      $form['webhook_event_type']['#description'] = $this->t('No webhook configurations found. Please create webhook configurations in the Mautic API module first.');
    }

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token Name'),
      '#description' => $this->t('Enter the name for the token that will contain the webhook data.'),
      '#default_value' => $this->configuration['token_name'] ?? 'webhook_data',
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['webhook_event_type'] = $form_state->getValue('webhook_event_type');
    $this->configuration['token_name'] = $form_state->getValue('token_name');
  }

  /**
   * {@inheritdoc}
   */
  public function getData(string $key): mixed {
    $event = $this->event;
    if ($event instanceof MauticWebhookReceived) {
      $token_name = $this->configuration['token_name'] ?? 'webhook_data';
      if ($key === $token_name) {
        return DataTransferObject::create($event->getData());
      }
    }
    return parent::getData($key);
  }

  /**
   * {@inheritdoc}
   */
  #[Token(
    name: 'webhook_data',
    description: 'Raw webhook data from Mautic.',
    classes: [
      MauticWebhookReceived::class,
    ],
    properties: [],
  )]
  protected function buildEventData(): array {
    $event = $this->event;
    $data = parent::buildEventData();
    if ($event instanceof MauticWebhookReceived) {
      $token_name = $this->configuration['token_name'] ?? 'webhook_data';
      $data[$token_name] = $event->getData();
    }
    return $data;
  }

}
