<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Action to update a contact in Mautic.
 *
 * @Action(
 *   id = "mautic_eca_update_contact",
 *   label = @Translation("Mautic: Update Contact"),
 *   description = @Translation("Update the contact for the selected Mautic connection.")
 * )
 */
class MauticUpdateContact extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mautic_id' => '',
      'json_data' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['mautic_id'] = [
      '#type' => 'text',
      '#title' => $this->t('Mautic ID'),
      '#description' => $this->t('Enter the mautic ID of the contact to update.'),
      '#default_value' => $this->configuration['mautic_id'],
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['json_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('JSON Data'),
      '#description' => $this->t('Enter JSON data to update the contact in Mautic. This field must contain valid JSON and can contain tokens.'),
      '#default_value' => $this->configuration['json_data'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $jsonDataString = $this->tokenService->replaceClear($this->configuration['json_data']);
      $jsonData = json_decode($jsonDataString, TRUE);
      $mauticContactId = $this->tokenService->replaceClear($this->configuration['mautic_id']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection);
      $response = $api->edit($mauticContactId, $jsonData);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully updated Mautic contact with ID: @id', [
          '@id' => $mauticContactId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Mautic contact: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
