<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Action to create a segment in Mautic.
 *
 * @Action(
 *   id = "mautic_eca_create_segment",
 *   label = @Translation("Mautic: Create Segment"),
 *   description = @Translation("Create a new segment in the selected Mautic instance.")
 * )
 */
class MauticCreateSegment extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'json_data' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['json_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('JSON Data'),
      '#description' => $this->t('Enter the JSON data to create the segment in Mautic. This field must contain valid JSON and can contain tokens.'),
      '#default_value' => $this->configuration['json_data'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['json_data'] = $form_state->getValue('json_data');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $json_data = $form_state->getValue('json_data');
    if (!empty($json_data)) {
      json_decode($json_data);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('json_data', $this->t('The JSON data is invalid: @error', ['@error' => json_last_error_msg()]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $jsonDataString = $this->tokenService->replaceClear($this->configuration['json_data']);
      $jsonData = json_decode($jsonDataString, TRUE);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      // Get the API object for segments.
      $api = $this->mauticApiConnector->getApi($connection, 'segments');

      // Create the segment using the API.
      $response = $api->create($jsonData);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully created Mautic segment with ID: @id', [
          '@id' => $response['segment']['id'],
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Mautic segment: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
