<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Action to list / find a segment in Mautic.
 *
 * @Action(
 *   id = "mautic_eca_list_segments",
 *   label = @Translation("Mautic: List Segments"),
 *   description = @Translation("List segments in the selected Mautic connection.")
 * )
 */
class MauticListSegments extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'search' => '',
      'start' => 0,
      'limit' => 30,
      'order_by' => '',
      'order_by_dir' => 'ASC',
      'published_only' => FALSE,
      'minimal' => FALSE,
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search filter'),
      '#description' => $this->t('String or search command to filter segments by. Supports tokens.'),
      '#default_value' => $this->configuration['search'],
    ];

    $form['start'] = [
      '#type' => 'number',
      '#title' => $this->t('Start'),
      '#description' => $this->t('Starting row for the segments returned.'),
      '#default_value' => $this->configuration['start'],
      '#min' => 0,
    ];

    $form['limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Limit'),
      '#description' => $this->t('Limit number of segments to return. 0 to return all.'),
      '#default_value' => $this->configuration['limit'],
      '#min' => 0,
    ];

    $form['order_by'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Order by'),
      '#description' => $this->t('Column to sort by. Can use any column listed in the response.'),
      '#default_value' => $this->configuration['order_by'],
    ];

    $form['order_by_dir'] = [
      '#type' => 'select',
      '#title' => $this->t('Sort direction'),
      '#description' => $this->t('Sort direction: asc or desc.'),
      '#default_value' => $this->configuration['order_by_dir'],
      '#required' => TRUE,
      '#options' => [
        'ASC' => $this->t('Ascending'),
        'DESC' => $this->t('Descending'),
      ],
    ];

    $form['published_only'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Published only'),
      '#description' => $this->t('Only return currently published segments.'),
      '#default_value' => $this->configuration['published_only'],
    ];

    $form['minimal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Minimal'),
      '#description' => $this->t('Return only array of segments without additional lists in it.'),
      '#default_value' => $this->configuration['minimal'] ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $search = $this->tokenService->replaceClear($this->configuration['search']);
      $start = $this->configuration['start'];
      $limit = $this->configuration['limit'];
      $order_by = $this->configuration['order_by'];
      $order_by_dir = $this->configuration['order_by_dir'];
      $published_only = $this->configuration['published_only'];
      $minimal = $this->configuration['minimal'];

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      // Get the API object for segments.
      $api = $this->mauticApiConnector->getApi($connection, 'segments');

      // List the segments using the API.
      $response = $api->getList($search, $start, $limit, $order_by, $order_by_dir, $published_only, $minimal);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully found Mautic segments');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Mautic segments: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['search'] = $form_state->getValue('search');
    $this->configuration['start'] = $form_state->getValue('start');
    $this->configuration['limit'] = $form_state->getValue('limit');
    $this->configuration['order_by'] = $form_state->getValue('order_by');
    $this->configuration['order_by_dir'] = $form_state->getValue('order_by_dir');
    $this->configuration['published_only'] = $form_state->getValue('published_only');
    $this->configuration['minimal'] = $form_state->getValue('minimal');

    parent::submitConfigurationForm($form, $form_state);
  }

}
