<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Action to add a contact to Do Not Contact in Mautic.
 *
 * @Action(
 *   id = "mautic_eca_add_do_not_contact",
 *   label = @Translation("Mautic: Add Contact to Do Not Contact"),
 *   description = @Translation("Add a contact to Do Not Contact in the selected Mautic connection.")
 * )
 */
class MauticAddDoNotContact extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'contact_id' => '',
      'channel' => 'email',
      'reason' => 3,
      'channel_id' => '',
      'comments' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['contact_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Contact ID'),
      '#description' => $this->t('Enter the Mautic contact ID.'),
      '#default_value' => $this->configuration['contact_id'],
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['channel'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel'),
      '#description' => $this->t("Channel of DNC. For example 'email', 'sms'. Default is 'email'."),
      '#default_value' => $this->configuration['channel'],
      '#required' => TRUE,
    ];

    $form['reason'] = [
      '#type' => 'select',
      '#title' => $this->t('Reason'),
      '#description' => $this->t('Reason for DNC. 1: Unsubscribed, 2: Bounced, 3: Manual.'),
      '#default_value' => $this->configuration['reason'],
      '#options' => [
        1 => $this->t('Unsubscribed'),
        2 => $this->t('Bounced'),
        3 => $this->t('Manual'),
      ],
      '#required' => TRUE,
    ];

    $form['channel_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel ID'),
      '#description' => $this->t('ID of the entity which was the reason for unsubscription (optional).'),
      '#default_value' => $this->configuration['channel_id'],
      '#required' => FALSE,
    ];

    $form['comments'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Comments'),
      '#description' => $this->t('A text describing details of DNC entry (optional).'),
      '#default_value' => $this->configuration['comments'],
      '#required' => FALSE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['contact_id'] = $form_state->getValue('contact_id');
    $this->configuration['channel'] = $form_state->getValue('channel');
    $this->configuration['reason'] = $form_state->getValue('reason');
    $this->configuration['channel_id'] = $form_state->getValue('channel_id');
    $this->configuration['comments'] = $form_state->getValue('comments');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $contactId = (int) $this->tokenService->replaceClear($this->configuration['contact_id']);
      $channel = $this->tokenService->replaceClear($this->configuration['channel']);
      $reason = (int) $this->tokenService->replaceClear($this->configuration['reason']);
      $channelId = $this->tokenService->replaceClear($this->configuration['channel_id']);
      $comments = $this->tokenService->replaceClear($this->configuration['comments']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection);
      $response = $api->addDnc($contactId, $channel, $reason, $channelId ?: NULL, $comments ?: 'via Drupal ECA');

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully added contact @id to Do Not Contact', [
          '@id' => $contactId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to add contact to Do Not Contact: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
