<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;

/**
 * Action to update a segment in Mautic.
 *
 * @Action(
 *   id = "mautic_eca_update_segment",
 *   label = @Translation("Mautic: Update Segment"),
 *   description = @Translation("Update the segment for the selected Mautic connection.")
 * )
 */
class MauticUpdateSegment extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mautic_id' => '',
      'json_data' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['mautic_id'] = [
      '#type' => 'text',
      '#title' => $this->t('Mautic ID'),
      '#description' => $this->t('Enter the Mautic ID of the segment to update.'),
      '#default_value' => $this->configuration['mautic_id'],
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['json_data'] = [
      '#type' => 'textarea',
      '#title' => $this->t('JSON Data'),
      '#description' => $this->t('Enter JSON data to update the segment in Mautic. This field must contain valid JSON and can contain tokens.'),
      '#default_value' => $this->configuration['json_data'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['mautic_id'] = $form_state->getValue('mautic_id');
    $this->configuration['json_data'] = $form_state->getValue('json_data');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $json_data = $form_state->getValue('json_data');
    if (!empty($json_data)) {
      json_decode($json_data);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName('json_data', $this->t('The JSON data is invalid: @error', ['@error' => json_last_error_msg()]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $jsonDataString = $this->tokenService->replaceClear($this->configuration['json_data']);
      $jsonData = json_decode($jsonDataString, TRUE);
      $mauticSegmentId = $this->tokenService->replaceClear($this->configuration['mautic_id']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection, 'segments');
      $response = $api->edit($mauticSegmentId, $jsonData);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully updated Mautic segment with ID: @id', [
          '@id' => $mauticSegmentId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Mautic segment: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
