<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Action that adds a contact to a segment in a specified Mautic instance.
 */
#[Action(
  id: 'mautic_eca_add_contact_to_segment',
  label: new TranslatableMarkup('Mautic: Add Contact to Segment'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Add a Contact to a Segment in a specified Mautic instance.'),
  version_introduced: '1.0.0',
)]
class MauticAddContactToSegment extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $segmentId = $this->tokenService->replaceClear($this->configuration['segment_id']);
      $contactId = $this->tokenService->replaceClear($this->configuration['contact_id']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection|null $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);
      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection, 'segments');

      $response = $api->addContact($segmentId, $contactId);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully added contact ID @contact_id to segment ID @segment_id', [
          '@contact_id' => $contactId,
          '@segment_id' => $segmentId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to add contact to segment: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'segment_id' => '',
      'contact_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['segment_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Segment ID'),
      '#description' => $this->t('Enter the Mautic Segment ID to add the contact to.'),
      '#default_value' => $this->configuration['segment_id'] ?? '',
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['contact_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Contact ID'),
      '#description' => $this->t('Enter the Mautic Contact ID to add to the segment.'),
      '#default_value' => $this->configuration['contact_id'] ?? '',
      '#required' => TRUE,
      '#min' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['segment_id'] = $form_state->getValue('segment_id');
    $this->configuration['contact_id'] = $form_state->getValue('contact_id');
    parent::submitConfigurationForm($form, $form_state);
  }

}
