<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Action to get a segment from Mautic.
 */
#[Action(
  id: 'mautic_eca_get_segment',
  label: new TranslatableMarkup('Mautic: Get Segment'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Get a segment from the selected Mautic connection.'),
)]
class MauticGetSegment extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'mautic_id' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['mautic_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Mautic ID'),
      '#description' => $this->t('Enter the Mautic ID of the segment to get.'),
      '#default_value' => $this->configuration['mautic_id'],
      '#required' => TRUE,
      '#min' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['mautic_id'] = $form_state->getValue('mautic_id');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $mauticSegmentId = $this->tokenService->replaceClear($this->configuration['mautic_id']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection|null $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection, 'segments');

      // Get the segment using the API.
      $response = $api->get($mauticSegmentId);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully retrieved the Mautic segment with the ID: @id', [
          '@id' => $mauticSegmentId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve the Mautic segment: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
