<?php

declare(strict_types=1);

namespace Drupal\mautic_eca\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Action to remove a contact from Do Not Contact in Mautic.
 */
#[Action(
  id: 'mautic_eca_remove_do_not_contact',
  label: new TranslatableMarkup('Mautic: Remove Contact from Do Not Contact'),
)]
#[EcaAction(
  description: new TranslatableMarkup('Remove a contact from Do Not Contact in the selected Mautic connection.'),
)]
class MauticRemoveDoNotContact extends MauticActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'contact_id' => '',
      'channel' => 'email',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['contact_id'] = [
      '#type' => 'number',
      '#title' => $this->t('Contact ID'),
      '#description' => $this->t('Enter the Mautic contact ID.'),
      '#default_value' => $this->configuration['contact_id'],
      '#required' => TRUE,
      '#min' => 1,
    ];

    $form['channel'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Channel'),
      '#description' => $this->t("Channel of DNC. For example 'email', 'sms'. Default is 'email'."),
      '#default_value' => $this->configuration['channel'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['contact_id'] = $form_state->getValue('contact_id');
    $this->configuration['channel'] = $form_state->getValue('channel');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    try {
      $connectionId = $this->configuration['mautic_connection'];
      $contactId = (int) $this->tokenService->replaceClear($this->configuration['contact_id']);
      $channel = $this->tokenService->replaceClear($this->configuration['channel']);

      /** @var \Drupal\mautic_api\Entity\MauticApiConnection|null $connection */
      $connection = $this->entityTypeManager
        ->getStorage('mautic_api_connection')
        ->load($connectionId);

      if (!$connection) {
        $this->logger->error('Mautic connection with ID @id not found', [
          '@id' => $connectionId,
        ]);
        return;
      }

      $api = $this->mauticApiConnector->getApi($connection);
      $response = $api->removeDnc($contactId, $channel);

      if ($response && !isset($response['errors'])) {
        $this->tokenService->addTokenData($this->configuration['token_name'], $response);
        $this->logger->info('Successfully removed contact @id from Do Not Contact', [
          '@id' => $contactId,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to remove contact from Do Not Contact: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
