<?php

declare(strict_types=1);

namespace Drupal\mcp\Plugin\McpJsonRpc;

use Drupal\jsonrpc\Object\ParameterBag;
use Drupal\mcp\Plugin\McpJsonRpcMethodBase;
use Drupal\mcp\ServerFeatures\Tool;

/**
 * Lists available tools from the server.
 *
 * @JsonRpcMethod(
 *   id = "tools/list",
 *   usage = @Translation("List available tools."),
 *   params = {
 *     "cursor" = @JsonRpcParameterDefinition(
 *       schema={"type"="string"},
 *       required=false,
 *       description=@Translation("An opaque token representing the current
 *       pagination position.")
 *     )
 *   }
 * )
 */
class ToolsList extends McpJsonRpcMethodBase {

  /**
   * {@inheritdoc}
   */
  public function execute(ParameterBag $params) {
    $tools = [];
    foreach ($this->mcpPluginManager->getAvailablePlugins() as $instance) {
      // Skip plugins that user doesn't have access to.
      if (!$instance->hasAccess()->isAllowed()) {
        continue;
      }

      $instanceTools = $instance->getToolsWithCustomization();
      $availableTools = [];

      // Filter tools based on access and enabled status.
      foreach ($instanceTools as $tool) {
        // Check if tool is enabled and user has access.
        if ($instance->isToolEnabled($tool->name) &&
            $instance->hasToolAccess($tool->name)->isAllowed()) {
          $availableTools[] = $tool;
        }
      }

      $prefixizedTools = array_map(
        fn($tool) => new Tool(
          name: $instance->generateToolId($instance->getPluginId(), $tool->name),
          description: $tool->description,
          inputSchema: $tool->inputSchema,
        ),
        $availableTools
      );
      $tools = array_merge($tools, $prefixizedTools);
    }

    return [
      'tools' => $tools,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function outputSchema() {
    return [
      'type'       => 'object',
      'required'   => ['tools'],
      'properties' => [
        'tools'      => [
          'type'  => 'array',
          'items' => [
            'type'       => 'object',
            'required'   => ['name', 'inputSchema'],
            'properties' => [
              'name'        => [
                'type'        => 'string',
                'description' => 'The name of the tool',
              ],
              'description' => [
                'type'        => 'string',
                'description' => 'A human-readable description of the tool',
              ],
              'inputSchema' => [
                'type'       => 'object',
                'required'   => ['type'],
                'properties' => [
                  'type' => ['const' => 'object'],
                ],
              ],
            ],
          ],
        ],
        'nextCursor' => [
          'type'        => 'string',
          'description' => 'Token for the next page of results',
        ],
      ],
    ];
  }

}
